
from logging import getLogger

from django.conf import settings
from django.dispatch import receiver
from ylog.context import log_context

from wiki.sync.connect.dao import get_operating_mode
from wiki.sync.connect.logic import disable_organization, make_sync_random_for_req_id
from wiki.sync.connect.signals import (
    domain_master_changed,
    organization_imported,
    organization_subscription_plan_changed,
)
from wiki.org import org_ctx, org_user
from wiki.pages.logic.init_org_data import init_org_data
from wiki.sync.connect.dir_client import DirClient, OrganizationAlreadyDeletedError
from wiki.users_biz.signals import get_dir_organization

logger = getLogger(__name__)


@receiver(organization_imported)
def process_imported_organization(sender, object, **kwargs):
    dir_client = DirClient(sync_random=make_sync_random_for_req_id(), timeout=30)
    with log_context(req_id=dir_client.get_req_id(object['id'])):
        try:
            dir_client.get_organization(object['id'])
        except OrganizationAlreadyDeletedError:
            logger.warn('Attempt to import deleted organization %s. Skipping rest of flow' % object['id'])
            disable_organization(object['id'])
            return

        init_org_data(object)
        enabled, ready = dir_client.get_service_status(object['id'])

        if enabled:
            dir_client.post_service_ready(
                service_slug=settings.DIRECTORY_SERVICE_SLUG,
                dir_org_id=object['id'],
            )
        else:
            logger.info('Service is not enabled, will not report that it is ready')


@receiver(domain_master_changed)
def process_domain_master_changed(sender, object, **kwargs):
    # При смене мастер-домена организации
    # у всех пользователей меняется почта.
    # Возможно, стоит подумать о том, чтобы
    # хранить мастер-домены в таблице организаций,
    # и составлять почту пользователя динамически.

    dir_org_id = str(object['org_id'])

    dir_client = DirClient(sync_random=make_sync_random_for_req_id())
    with log_context(req_id=dir_client.get_req_id(dir_org_id)):
        org_obj = dir_client.get_organization(dir_org_id, fields='domains')

    master_domain = org_obj['domains']['master']

    logger.info('Master-domain for org %s changed to "%s", updating users emails', dir_org_id, master_domain)

    org = get_dir_organization(dir_org_id)
    with org_ctx(org):
        for user in org_user():
            email = '{}@{}'.format(user.username, master_domain)
            user.email = email
            user.save()
            user.staff.work_email = email
            user.staff.save()


@receiver(organization_subscription_plan_changed)
def process_organization_subscription_plan_changed(sender, object, content, **kwargs):
    """
    Обработать сигнал об изменении тарифного плана организации.
    """

    dir_org_id = str(object['id'])
    plan = content['subscription_plan']
    org = get_dir_organization(dir_org_id)
    with org_ctx(org):
        mode = get_operating_mode(plan)
        old_org_mode = org.mode.name
        if old_org_mode == mode.name:
            logger.warn(
                'Attempt to change the Subscription plan of organization %s to the same: %s', dir_org_id, old_org_mode
            )
        else:
            org.mode = mode
            org.save()
            logger.info(
                "Subscription plan of organization %s was changed from '%s' to '%s'",
                dir_org_id,
                old_org_mode,
                mode.name,
            )
