from typing import List, Optional

from django.db import models
from mptt.querysets import TreeQuerySet

from wiki.acl.consts import AclSubjectType, AclSubject
from wiki.intranet.models.intranet_extensions import Group as DISGroup
from wiki.users.abstract_user import AbstractWikiUser


class ActiveGroupManager(models.Manager):
    def get_queryset(self):
        return super(ActiveGroupManager, self).get_queryset().filter(intranet_status__gt=0)


class Group(DISGroup):
    active = ActiveGroupManager()

    class Meta:
        proxy = True
        app_label = 'users_ynx'

    def get_all_members(self):
        return User.objects.filter(staff__in=super(Group, self).get_all_members())

    def get_all_staff_members(self):
        return super(Group, self).get_all_members()

    def get_all_members_count(self):
        return super(Group, self).get_all_members().count()

    def get_acl_subject(self):
        return AclSubject(subj_id=self.id, subj_type=AclSubjectType.GROUP)


class User(AbstractWikiUser):
    """
    Модель Вики пользователя, использующаяся в инстансах Вики для Яндекса.
    """

    def _get_all_groups_queryset(self) -> TreeQuerySet:
        first_level_groups_qs = self.staff.in_groups.filter(intranet_status=1)
        return DISGroup.tree.get_queryset_ancestors(first_level_groups_qs, include_self=True)

    def get_all_groups(self) -> List[DISGroup]:
        return list(self._get_all_groups_queryset())

    def get_all_group_ids(self) -> List[int]:
        return list(self._get_all_groups_queryset().values_list('id', flat=True))

    class Meta(AbstractWikiUser.Meta):
        app_label = 'users_ynx'

    def set_cloud_uid(self, cloud_uid: str):
        pass

    def get_cloud_uid(self) -> Optional[str]:
        return None

    def set_uid(self, cloud_uid: str):
        pass

    def get_uid(self) -> str:
        return self.staff.uid

    # поле для изоморфности, если кто-то запросит напрямую без функций выше
    @property
    def cloud_uid(self):
        return None
