import logging
import time
import uuid

import requests

from wiki.utils.errors import (
    ApiRequestBadRequest,
    ApiRequestFailed,
    ApiRequestForbidden,
    ApiRequestNotFound,
    ApiRequestWorkerFatal,
)
from ylog.context import log_context

logger = logging.getLogger(__name__)

MAX_RETRIES = 3


def api_request(
    method,
    url,
    service_ticket=None,
    user_ticket=None,
    oauth_token=None,
    headers=None,
    retry_500=False,
    remote_name=None,
    max_retries=None,
    **kwargs,
):
    req_id = str(uuid.uuid4())

    max_retries = max_retries or MAX_RETRIES

    if not headers:
        headers = {}

    if service_ticket:
        headers['X-Ya-Service-Ticket'] = service_ticket
    if user_ticket:
        headers['X-Ya-User-Ticket'] = user_ticket
    if oauth_token:
        headers['Authorization'] = 'OAuth %s' % oauth_token

    headers['X-Req-Id'] = req_id

    attempt = 0
    fallback_dur = 0.5

    prompt = f'[ApiCall {remote_name}]'
    last_error = ''
    with log_context(api_url=url, api_method=method, api_req_id=req_id):
        while attempt < max_retries:
            attempt += 1

            try:
                response = requests.request(method, url, headers=headers, **kwargs)
                logline = f'{prompt} Api request {attempt}/{MAX_RETRIES}; status {response.status_code}'
                last_error = f'server status code is {response.status_code}'

                if 200 <= response.status_code <= 299:
                    if response.status_code != 200:
                        logger.info(logline)
                    return response

                elif response.status_code == 500:  # - fatal statuses
                    logger.warning(logline)
                    if not retry_500:
                        raise ApiRequestWorkerFatal()

                else:
                    logger.warning(logline)
                    if response.status_code == 400:
                        raise ApiRequestBadRequest(response.content)
                    elif response.status_code == 404:
                        raise ApiRequestNotFound()
                    if response.status_code == 409:
                        raise ApiRequestBadRequest(response.content)
                    elif response.status_code in {401, 403}:
                        raise ApiRequestForbidden(response.content)
                    elif 400 < response.status_code < 429:
                        raise ApiRequestFailed()  # нет смысла пробовать еще раз, сервис говорит, что наш запрос плохой

            except (requests.exceptions.Timeout, requests.exceptions.RequestException) as e:
                last_error = str(e)
                logger.warning(f'{prompt} Api request failed, will retry ({str(e)})')

            time.sleep(fallback_dur)
            fallback_dur *= 2

        logger.error(f'{prompt} Out of retries -- {last_error}')
    raise ApiRequestFailed()


def get(url, **kwargs):
    return api_request('GET', url, **kwargs)


def post(url, **kwargs):
    return api_request('POST', url, **kwargs)


def put(url, **kwargs):
    return api_request('PUT', url, **kwargs)


def patch(url, **kwargs):
    return api_request('PATCH', url, **kwargs)


def delete(url, **kwargs):
    return api_request('DELETE', url, **kwargs)
