import json

import kikimr.public.sdk.python.persqueue.grpc_pq_streaming_api as pqlib
from kikimr.public.sdk.python.persqueue.errors import SessionFailureResult


class LogbrokerClient:
    """Клиент для логброкера.

    with LogbrokerClient() as logbroker:
        ...

    """

    timeout = 15

    def __init__(self, creds='', datacenter='myt', timeout=None):
        """
        :param token: Токен доступа. Если не передан, будет взят из переменной окружения LOGBROKER_TOKEN
        :param datacenter: man, vla, myt ...
        :param timeout:
        """

        self._api = None
        self.timeout = timeout or self.timeout
        self.endpoint = '%s.logbroker.yandex.net' % datacenter

        self.credentials = creds

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.quit()

    def quit(self):
        api = self._api
        api and api.stop()

    def _init(self):
        api = pqlib.PQStreamingAPI(self.endpoint, 2135)
        api_start_future = api.start()
        api_start_future.result(timeout=self.timeout)
        self._api = api

        return api

    @property
    def api(self):
        api = self._api

        if not api:
            api = self._init()

        return api

    def get_producer(self, source, topic):
        """Возвращает писателя в определённый раздел.

        :param source: Идентификатор, описывающий источник. Произвольная строка.
        :param topic: Раздел, куда требуется производить запись.

        """
        return LogbrokerProducer(client=self, source=source, topic=topic)


class LogbrokerProducer:
    """Писчий в логброкер.

    with LogbrokerClient() as logbroker:

        with logbroker.get_producer(source='me', topic='some/there/here') as producer:
            producer.write({'some': 'value'})
            producer.write('plaintext')

    """

    timeout = 10

    def __init__(self, client, source, topic, timeout=None):

        self.timeout = timeout or self.timeout
        self.client = client
        self.seq_num = 0
        self.topic = topic
        self.source = source
        self._me = None

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.close()

    def close(self):
        me = self._me
        me and me.stop()

    def _init(self):
        client = self.client
        configurator = pqlib.ProducerConfigurator(self.topic.encode(), self.source.encode())
        producer = client.api.create_producer(configurator, credentials_provider=client.credentials)
        start_future = producer.start()
        result = start_future.result(timeout=self.timeout)
        if isinstance(result, SessionFailureResult) or not result.HasField('init'):
            raise AssertionError('Unable to initialize Logbroker producer: %s' % result)
        self.seq_num = result.init.max_seq_no
        self._me = producer

        return producer

    def write(self, message):
        producer = self._me

        if not producer:
            producer = self._init()
        self.seq_num += 1

        seq_num = self.seq_num

        if isinstance(message, dict):
            message = json.dumps(message)

        if isinstance(message, str):
            message = message.encode()
        response = producer.write(seq_num, message)
        result = response.result(timeout=self.timeout)
        if not result.HasField('ack'):
            raise AssertionError('Unable to write Logbroker message: %s' % result)
        return seq_num
