
import logging
import threading
import time
from argparse import ArgumentTypeError
from datetime import datetime as dt

from django_mds.storage import MDSStorage

#
# Loggers
#

command_logger = logging.getLogger('wiki.management_commands')
task_logger = logging.getLogger('celery_task')

#
# Script helpers
#

percents = threading.RLock()  # percents computing lock


def print_percents(processed, count_struct, every=100, logger=None):
    """
    Print processed percents thread-safely

    @type processed: int
    @param processed: newly processed objects count
    @type count_struct: dict
    @param count_struct: {'current': <currently_processed>, 'all': <total_to_process>}
    @param logger: script logger
    @rtype: int
    @return: processed count back
    """

    if processed % every == 0:
        percents.acquire()
        count_struct['current'] += processed
        percent = round(float(count_struct['current']) * 100 / count_struct['all'], 2)
        logger.info(
            'Processed {0}% (this thread: "{1}", all threads "{2}" of {3})'.format(
                percent, processed, count_struct['current'], count_struct['all']
            )
        )
        percents.release()

    return processed


def get_storage(read_url, write_url, access_token):
    """
    Return Storage for url.
    """
    # note that MDS_NAMESPACE will be taken from instance settings module
    return MDSStorage(host=read_url, write_host=write_url, write_token=access_token, read_token=access_token)


def spawner(target=None, args=None, maxthreads=0):
    """
    Spawn number of threads with arguments

    @type target: callable
    @param target: callable target for each thread
    @param args: arguments for each thread
    @param maxthreads: maximum number of threads
    """
    initial_thread_count = threading.active_count()

    threads = [threading.Thread(target=target, name='loader' + str(index), args=args) for index in range(maxthreads)]

    # starting threads as daemonic threads so they'll exit
    # on KeyboardInterrupt in main thread
    for thread in threads:
        thread.daemon = True
        thread.start()

    # wait thread until all threads are done running
    # when all done, only main thread must remain
    while threading.active_count() > initial_thread_count:
        time.sleep(1)

    for thread in threads:
        thread.join()


def argparse_datetime_type(value):
    datetime_format = '%Y-%m-%d %H:%M:%S'
    try:
        return dt.strptime(value, datetime_format)
    except ValueError:
        raise ArgumentTypeError('{} is not a valid value. Provide datetime in {} format'.format(value, datetime_format))
