import logging
import socket
from datetime import datetime
from dateutil import parser
from dataclasses import dataclass
from typing import List, Union
from enum import Enum

from blackbox import JsonBlackbox
from django.conf import settings
from wiki.utils.tvm2 import get_service_ticket

logger = logging.getLogger('wiki.utils.blackbox')

blackbox = JsonBlackbox(url=settings.BLACKBOX_URL)

try:
    SERVICE_IP = socket.gethostbyname(socket.gethostname())
except:  # noqa
    SERVICE_IP = '127.0.0.1'


class BlackboxResponseStatus(Enum):
    VALID = 'VALID'
    INVALID = 'INVALID'


class ExpiredTokenError(Exception):
    pass


class BlackboxError(Exception):
    pass


@dataclass
class AvatarData:
    default: str = ''
    empty: bool = True


@dataclass
class UserData:
    id: str
    avatar: AvatarData
    uid: str = ''
    login: str = ''


def get_users_data_by_uids(uids: List[str]) -> List[UserData]:
    bb_response = blackbox.userinfo(
        uid=uids,
        userip=SERVICE_IP,
        regname='yes',
        headers={'X-Ya-Service-Ticket': get_service_ticket(settings.TVM2_BLACKBOX_CLIENT.value)},
    )

    users_data = bb_response.get('users', [])
    if not users_data:
        logger.warning('Can\'t get users info from blackbox, response: "%s"', bb_response)

    result = []
    for user_data in users_data:
        uid_data = user_data['uid']
        if uid_data:
            data = UserData(
                id=user_data['id'],
                uid=uid_data['value'],
                login=user_data['login'],
                avatar=AvatarData(
                    default=user_data['display_name']['avatar']['default'],
                    empty=user_data['display_name']['avatar']['empty'],
                ),
            )
        else:
            data = UserData(id=user_data['id'], avatar=AvatarData())
        result.append(data)

    return result


def get_oauth_token_expire_time(token: str) -> Union[datetime, None]:
    """
    Возвращает дату окончания срока жизни токена или None, если срок жизни неограничен
    """
    bb_response = blackbox.oauth(
        oauth_token=token,
        userip=SERVICE_IP,
        headers={'X-Ya-Service-Ticket': get_service_ticket(settings.TVM2_BLACKBOX_CLIENT.value)},
    )
    status = bb_response['status']
    if status['value'] == BlackboxResponseStatus.INVALID.value:
        error = bb_response.get('error')
        if error == 'expired_token':
            raise ExpiredTokenError()
        else:
            raise BlackboxError(error)

    date_str = bb_response['oauth']['expire_time']
    return parser.parse(date_str) if date_str else None
