# coding=utf-8

import logging

from django.core.cache import BaseCache, caches
from django.core.cache.backends.base import DEFAULT_TIMEOUT


def load_cache(desc):
    """
    Return cache backend using name or definition.:TwoLevelCache
    """
    if isinstance(desc, str):
        return caches[desc]
    else:
        alias = desc['alias']
        return caches[alias]


MISSING = object()


class TwoLevelCache(BaseCache):
    """
    Two-level cache. Look for key in the first one, then in the second one.

    Second level acts as real, reliable cache. First level is supposed to be
    fast, but can be incorrect in some cases.

    Be very careful with the first level! It does not get invalidated
    properly. You should keep it as short-lived as possible, or all sorts of
    weirdness can occur.
    """

    def __init__(self, location, params):
        first = params['first']
        second = params['second']
        # If set to True, recache level 2 misses - this is useful if level 2
        # is expensive to use.
        self.recache_misses = params.get('RECACHE_MISSES', False)

        self.first = load_cache(first)
        self.second = load_cache(second)

    def set(self, key, value, timeout=DEFAULT_TIMEOUT, version=None):
        self.second.set(key, value, timeout, version)
        self.first.set(key, value, timeout, version)

    def get(self, key, default=None, version=None, second_level_only=False):
        """
        Сигнатура отличается от родительской параметром second_level_only.
        """
        if not second_level_only:
            result = self.first.get(key, version=version, default=MISSING)
            if result is not MISSING:
                logging.debug('Got result from first level cache')
                return result
            else:
                logging.debug('Object is MISSING (is "%s") in first level cache', repr(result))
        else:
            logging.debug('Using only second level cache')

        result = self.second.get(key, version=version, default=MISSING)

        if (result is not MISSING) or self.recache_misses:
            logging.debug('Recache second level cache')
            self.first.set(key, result, version=version)
        else:
            result = default

        logging.debug('Return result from second level cache: "%s"', repr(result))
        return result

    def delete(self, key, version=None):
        self.first.delete(key, version=version)
        self.second.delete(key, version=version)

    def clear(self):
        self.first.clear()
        self.second.clear()
