"""
    This module provides decorator @callsuper. Methods defined with callsuper
    automatically invoke super methods if they exist. See tests to
    understand usage.
"""

from functools import partial, wraps


class CallsuperError(Exception):
    pass


def callsuper(method=None, **kwargs):
    def decorator(method, **kwargs):
        deco_kwargs = kwargs

        @wraps(method)
        def wrapper(self, *args, **kwargs):

            # Find class that possesses wrapper
            for cls in self.__class__.__mro__:
                if wrapper in list(cls.__dict__.values()):
                    break

            # Call wrapped method BEFORE super method if instructed so
            if deco_kwargs.get('at_the_end'):
                result = method(self, *args, **kwargs)

            # Find and call super method
            super_self = super(cls, self)
            super_bound_method = getattr(super_self, method.__name__, None)
            if super_bound_method:
                s_args, s_kwargs = args[:], kwargs.copy()

                arg_modifier = getattr(wrapper, 'arg_modifier', None)
                if arg_modifier:
                    s_args, s_kwargs = arg_modifier(*s_args, **s_kwargs)

                super_result = super_bound_method(*s_args, **s_kwargs)
            else:
                super_result = None

            # Pass result of super method if instructed
            if 'result' in deco_kwargs:
                if isinstance(deco_kwargs['result'], str):
                    kwargs[deco_kwargs['result']] = super_result
                else:
                    kwargs['result'] = super_result

            # Call wrapped method
            if not deco_kwargs.get('at_the_end'):
                result = method(self, *args, **kwargs)

            return result

        def superargs(arg_modifier):
            wrapper.arg_modifier = arg_modifier
            return wrapper

        wrapper.superargs = superargs
        return wrapper

    if method is None:
        return partial(decorator, **kwargs)
    elif callable(method):
        return decorator(method, **kwargs)
    else:
        message = 'first nonkeyword arg should be method, not {0!r}'.format(method)
        raise CallsuperError(message)
