from os import path, remove
from subprocess import Popen, PIPE
from hashlib import sha1
from random import random
from difflib import SequenceMatcher
import re
from itertools import groupby

# Это разные с точки зрения алгоритмов константы
STRINGS_COMPARISON_THRESHOLD = 0.5  # чем меньше, тем больше должны совпадать строки для сравнения по словам
WORDS_COMPARISON_THRESHOLD = 0.7  # чем больше, тем больше должны совпадать слова для сравнения по буквам


def simple_diff(text1, text2):
    """
    Diff two texts by lines.
    E.g. text1 is a new version of text and text2 is an old one.
    """
    a = text1.splitlines() if text1 else []
    b = text2.splitlines() if text2 else []
    deletions = []
    additions = []
    for group in SequenceMatcher(None, a, b).get_grouped_opcodes():
        i1, i2, j1, j2 = group[0][1], group[-1][2], group[0][3], group[-1][4]
        for tag, i1, i2, j1, j2 in group:
            if tag == 'equal':
                continue
            if tag == 'replace' or tag == 'delete':
                for line in a[i1:i2]:
                    deletions.append(line)
            if tag == 'replace' or tag == 'insert':
                for line in b[j1:j2]:
                    additions.append(line)
    return (additions, deletions)


class GitDiff(object):
    addition = {'start': '<ins>', 'end': '</ins>'}
    deletion = {'start': '<del>', 'end': '</del>'}

    @property
    def _tmp_file_name(self):
        while True:
            tmp_name = path.join('/tmp', sha1(str(random()).encode()).hexdigest())
            if not path.exists(tmp_name):
                return tmp_name

    def _post_process(self, text):
        # strip headers
        text = text.decode()
        head_end_pos = text.find('\x1b[m', text.find('m@@ '))

        # split by console control symbol
        chunks = text[head_end_pos + 4 :].split('\x1b')
        state = None

        for i in range(len(chunks)):
            pos = chunks[i].find('m')
            # defense from false markers "m" in text
            if not chunks[i] or chunks[i][0] != '[':
                continue
            # control symbols can be use together separated by ;
            ctrls = chunks[i][1:pos].split(';')

            # 32 means green (addition)
            is_green = '32' in ctrls
            # 31 means red (deletion)
            is_red = '31' in ctrls
            if is_green or is_red:
                if is_green:
                    state = 'addition'
                elif is_red:
                    state = 'deletion'
                if state:
                    chunks[i] = getattr(self, state)['start'] + chunks[i][pos + 1 :]

            # empty control symbol means end of previous control
            elif not ctrls[0] and state:
                chunks[i] = getattr(self, state)['end'] + chunks[i][pos + 1 :]
                state = None

            else:
                chunks[i] = chunks[i][pos + 1 :]

        return ''.join(chunks)

    def diff_texts(self, text1, text2):
        size1 = len(text1)
        size2 = len(text2)
        file1 = self._tmp_file_name
        open(file1, 'wb').write(text1.strip().encode('utf-8') + b'\n')
        file2 = self._tmp_file_name
        open(file2, 'wb').write(text2.strip().encode('utf-8') + b'\n')
        res_diff = self.diff_files(file1, file2, size1 if size1 > size2 else size2)
        remove(file1)
        remove(file2)
        return res_diff

    def diff_files(self, file1, file2, lines=None):
        if lines is None:
            lines1 = Popen(('wc', '-l', file1)).communicate()[0].split(' ')[0]
            lines2 = Popen(('wc', '-l', file2)).communicate()[0].split(' ')[0]
            lines = lines1 if lines1 > lines2 else lines2
        cmd = ['git', 'diff', '--color-words', '-U' + str(lines), '--', file1, file2]
        res_diff = Popen(cmd, stdout=PIPE).communicate()[0]
        return self._post_process(res_diff).strip()


# Negative lookahead-блок: точка, находящаяся в середине слова (например, "yandex.ru") не будет разделителем слов,
# а точка перед пробелом или в конце строки будет.
SPACES_AND_AUX = re.compile(r'([\s\(\)\[\]\+=«»“”…]+)|([\.,\-:]+(?![^\s]))')
SPACES = re.compile(r'([\s]+)')
sjoin = ''.join


def word_split(string):
    words = [w for w in SPACES_AND_AUX.split(string) if w]
    # words может содержать знаки препинания, объединенные с пробелами в одном элементе.
    words = [s for w in words for s in SPACES.split(w) if s]
    return words


class DifflibDiff(object):
    def __init__(self, return_type='html', unchanged_lines_count=None):
        """
        @param return_type: 'html' | 'chunks'
        @param unchanged_lines_count: int | None

        Если return_type == 'html', то метод diff_text() будет возвращать текст со вставками тегов <ins> и <del>.
        Для этого необходимо предварительно эскейпить исходные тексты.
        Если return_type == 'chunks', то вернет массив строк, каждая строка представляет собой массив 2-таплов из
        операции ('+', '-' или '=') и подстроки.
        unchanged_lines_count: количество неизмененных строк, которые надо показывать рядом с измененными. None -
        показывать всe строки
        """
        self.return_type = return_type
        self.unchanged_lines_count = unchanged_lines_count

    def _diff_words(self, word_a, word_b):
        """Возвращает генератор различий между двумя словами.

        >>> d = DifflibDiff()

        Определим для разворачивания генератора функцию diff():
        >>> diff = lambda a, b: list(d._diff_words(a, b))

        >>> diff('foo', 'foo')
        [('=', 'foo')]
        >>> diff('hello', 'helo')
        [('=', 'hel'), ('-', 'l'), ('=', 'o')]
        >>> diff('hell', 'hello')
        [('=', 'hell'), ('+', 'o')]
        >>> diff('hallo', 'hello')
        [('=', 'h'), ('-', 'a'), ('+', 'e'), ('=', 'llo')]

        В случае отличия более, чем на 30%, и в случае более одного удаления/добавления/замены,
        считает слова разными:

        >>> diff('melon', 'hello')
        [('-', 'melon'), ('+', 'hello')]
        >>> diff('muscles', 'mussels')
        [('-', 'muscles'), ('+', 'mussels')]

        """
        word_matcher = SequenceMatcher(None, word_a, word_b, autojunk=False)

        if word_matcher.ratio() < WORDS_COMPARISON_THRESHOLD:
            yield '-', word_a
            yield '+', word_b
            return

        result = []
        for tag, i, j, k, l in word_matcher.get_opcodes():
            part_a = word_a[i:j]
            part_b = word_b[k:l]
            if tag == 'equal':
                result.append(('=', part_a))
            elif tag == 'replace':
                result.append(('-', part_a))
                result.append(('+', part_b))
            elif tag == 'insert':
                result.append(('+', part_b))
            elif tag == 'delete':
                result.append(('-', part_a))

        additions_count = sum(1 for o, l in result if o == '+')
        deletions_count = sum(1 for o, l in result if o == '-')
        replaces_count = sum(1 for i in range(len(result) - 1) if set([result[i][0], result[i + 1][0]]) == set('+-'))

        if (
            additions_count > 1
            or deletions_count > 1
            or (
                # Одна вставка и удаление, и они не рядом (то есть не одно изменение)
                additions_count == 1
                and deletions_count == 1
                and replaces_count == 0
            )
        ):
            yield '-', word_a
            yield '+', word_b
            return
        for chunk in result:
            yield chunk

    def _diff_lines(self, line_a, line_b):
        """Возвращает генератор различий между двумя строчками.

        >>> d = DifflibDiff()

        Т.к. метод возвращает генератор, определим для тестов функцию diff()
        >>> diff = lambda a, b: list(d._diff_lines(a, b))

        >>> diff('foo bar asd', 'bar asd baz')
        [('-', 'foo '), ('=', 'bar asd'), ('+', ' baz')]

        Если в различающихся группах разное количество слов, считать
        всю группу отличающейся:
        >>> diff('There is a green box here', 'There is a big red fox here')
        [('=', 'There is a '), ('-', 'green box'), ('+', 'big red fox)', ('=', ' here')]

        """
        line_a, line_b = word_split(line_a), word_split(line_b)

        line_matcher = SequenceMatcher(lambda s: s.isspace(), line_a, line_b, autojunk=False)

        for tag, i, j, k, l in line_matcher.get_opcodes():
            do_del_ins = False
            part_a = line_a[i:j]
            part_b = line_b[k:l]
            if tag == 'equal':
                yield '=', sjoin(part_a)
            if tag == 'replace':
                # Если количество слов совпадает и не превышает 3, пробуем сравнить попарно:
                if len(part_a) == len(part_b) < 4:
                    for word_a, word_b in zip(part_a, part_b):
                        for chunk in self._diff_words(word_a, word_b):
                            yield chunk
                else:
                    do_del_ins = True
            if tag == 'delete' or do_del_ins:
                yield '-', sjoin(part_a)
            if tag == 'insert' or do_del_ins:
                yield '+', sjoin(part_b)

    def _smart_diff_texts(self, distances_cache, text_a, text_b):
        """
        "Умное" различие между текстами. Пытается найти наиболее похожие строки.
        """
        closest_strings_indices = closest_strings(distances_cache, text_a, text_b)

        if closest_strings_indices is None:  # Все строчки сильно отличаются
            for line in text_a:
                yield [('-', line)]
            for line in text_b:
                yield [('+', line)]

        else:
            i, j = closest_strings_indices
            for result in self._smart_diff_texts(distances_cache, text_a[:i], text_b[:j]):
                yield result

            # Сравниваем строки. _diff_lines может вернуть несколько однотипных операций подряд, поэтому сгруппируем их
            line_diff = []
            for operation, chunk in groupby(self._diff_lines(text_a[i], text_b[j]), lambda o_c: o_c[0]):
                line_diff.append((operation, sjoin(c[1] for c in chunk)))
            yield line_diff

            for result in self._smart_diff_texts(distances_cache, text_a[(i + 1) :], text_b[(j + 1) :]):
                yield result

    def _diff_texts(self, text_a, text_b):
        r"""Возвращает генератор различий между двумя текстами.
        Первая итерация - строки
        Вторая итерация - tuple из операции (+, - или =) и подстроки

        >>> d = DifflibDiff()

        Т.к. метод возвращает генератор, определим для тестов функцию diff()
        >>> diff = lambda a, b: list(d._diff_lines(a, b))

        >>> diff(
        ...    'foo\nbook\nbaz\n',
        ...    'foo\nboom\nbaz\n',
        ... )
        [[('=', 'foo')], [('=', 'boo'), ('-', 'k'), ('+', 'm')], [('=', 'baz')]

        """
        text_a, text_b = text_a.split('\n'), text_b.split('\n')

        text_matcher = SequenceMatcher(None, text_a, text_b, autojunk=False)

        for tag, i, j, k, l in text_matcher.get_opcodes():
            part_a = text_a[i:j]
            part_b = text_b[k:l]
            if tag == 'equal':
                for line in part_b:
                    yield [('=', line)]
            if tag == 'replace':
                # Пробуем найти максимальную совпадающую строку (по нормированному расстоянию Левенштейна):
                # Строки будут сравниваться попарно не один раз, поэтому для каждого блока сравниваемых строк
                # будем хранить попарные расстояния, чтобы не считать много раз
                distances_cache = {}
                for result in self._smart_diff_texts(distances_cache, part_a, part_b):
                    yield result
            if tag == 'delete':
                for line in part_a:
                    yield [('-', line)]
            if tag == 'insert':
                for line in part_b:
                    yield [('+', line)]

    def diff_texts(self, text1, text2):
        r"""Возвращает различия между двумя текстами.

        >>> d = DifflibDiff()

        >>> print d.diff_texts('foo\nbo\nom\nbaz\n\n', 'foo\nboom\nbaz\n\n')
        foo
        <del>bo</del>
        <del>om</del>
        <ins>boom</ins>
        baz
        <BLANKLINE>
        <BLANKLINE>

        """
        changes = list(self._diff_texts(text1, text2))
        changes = self._filter_unchanged_lines(changes)

        if self.return_type == 'chunks':
            return changes

        result = []
        for line in changes:
            if line and line[0][0] == '.':  # Текущая строчка - плейсхолдер для пропущенных неизмененных строк
                result.append('...')
                continue

            line_result = []
            for operation, chunk in line:
                line_result.append(
                    '<ins>%s</ins>' % chunk
                    if operation == '+'
                    else '<del>%s</del>' % chunk
                    if operation == '-'
                    else chunk
                )
            result.append(sjoin(line_result))
        return '\n'.join(result)

    def _filter_unchanged_lines(self, changes):
        """
        Отфильтровать неизмененные строки из диффа при установленном параметре unchanged_lines_count
        """

        if self.unchanged_lines_count is None:
            return changes

        indices = []  # Индексы строк, которые должны быть включены в дифф
        last_indices = []  # Вспомогательный массив
        unchanged_lines_count = self.unchanged_lines_count
        for i, line in enumerate(changes):
            is_changed = any(o in '+-' for o, c in line)

            if is_changed:
                # Добавить строки перед измененной
                indices.extend(last_indices[-self.unchanged_lines_count :])
                last_indices = []

                # Добавить саму измененную строку
                unchanged_lines_count = self.unchanged_lines_count
                indices.append(i)

            else:
                if unchanged_lines_count:
                    # Если последняя измененная строка была "недавно", добавить текущую строку
                    unchanged_lines_count -= 1
                    indices.append(i)
                else:
                    # Добавить текущую строку в стек на случай, если "скоро" будет измененная строка
                    last_indices.append(i)

        indices.extend(last_indices[-self.unchanged_lines_count :])

        last_index = -1
        filtered_changes = []

        for index in indices:
            if last_index + 1 != index:
                filtered_changes.append([('.', '')])
            filtered_changes.append(changes[index])
            last_index = index

        return filtered_changes


def levenshtein_distance(s1, s2):
    """
    Расстояние Левенштейна между строками
    @param s1: str
    @param s2: str
    @return: int
    """

    if len(s1) < len(s2):
        s1, s2 = s2, s1

    # len(s1) >= len(s2)
    if len(s2) == 0:
        return len(s1)

    previous_row = range(len(s2) + 1)
    for i, c1 in enumerate(s1):
        current_row = [i + 1]
        for j, c2 in enumerate(s2):
            insertions = previous_row[j + 1] + 1
            deletions = current_row[j] + 1
            substitutions = previous_row[j] + (c1 != c2)
            current_row.append(min(insertions, deletions, substitutions))
        previous_row = current_row

    return previous_row[-1]


def weighted_levenstein_distance(s1, s2):
    return float(levenshtein_distance(s1, s2)) / max(len(s1), len(s2))


def prefix_suffix_distance(s1, s2):
    """
    Возвращает сумму длин общего префикса и суффикса.
    Отдается со знаком "-" (чем больше похожи строки, тем меньше расстояния).
    @param s1: basestring
    @param s2: basesting
    @return: basestring
    """
    return -(len(path.commonprefix([s1, s2])) + len(path.commonprefix([''.join(reversed(s1)), ''.join(reversed(s2))])))


def closest_strings(distances_cache, strings1, strings2):
    """
    strings1 и strings2 - массивы строк. Функция ищет две максимально похожие строки среди них. Если похожих нет,
    возвращает None. Если строк слишком много, также возвращает None.
    @param distances_cache: dict((str, str) -> float)
    @param strings1: list(str)
    @param strings2: list(str)
    @return: (int, int)
    """
    if max(len(strings1), len(strings2)) > 15:
        return

    result, min_distance = None, 1000000
    for i, string1 in enumerate(strings1):
        for j, string2 in enumerate(strings2):
            if string1 and string2:  # Пустые строки не сравниваем
                distance = distances_cache.get((string1, string2), prefix_suffix_distance(string1, string2))
                distances_cache[string1, string2] = distance
                if distance < min_distance:
                    result, min_distance = (i, j), distance

    if result:
        # Если расстояние Левенштейна у кандитата слишком маленькое, считаем, что совпадений нет
        i, j = result
        if weighted_levenstein_distance(strings1[i], strings2[j]) > STRINGS_COMPARISON_THRESHOLD:
            return

    return result
