"""Logging Handlers that send messages in ErrorBooster Log Format"""

import datetime
import http.client as httplib
import json
import logging
import os
import traceback

from wiki.utils.instance_properties import get_version
from ylog.context import get_log_context


def object_to_json(obj) -> str:
    if isinstance(obj, datetime.datetime):
        return obj.isoformat()
    return repr(obj)


class HTTPHandler(logging.Handler):
    def __init__(self, host: str, port: int, path='/errorbooster', timeout: float = 0.5):
        super().__init__()

        self.host = host
        self.port = port
        self.path = path
        self.timeout = timeout

    def emit(self, record: logging.LogRecord):
        if not self.host:
            return

        pickle = self.makePickle(record)

        try:
            connection = httplib.HTTPConnection(host=self.host, port=self.port, timeout=self.timeout)
            connection.request('POST', self.path, pickle)
        except Exception:
            pass

    def makePickle(self, record: logging.LogRecord) -> str:
        log_dict = self._make_log_dict(record)
        return json.dumps(log_dict, default=object_to_json)

    def _make_log_dict(self, record: logging.LogRecord) -> dict:
        from django.conf import settings  # noqa

        log_dict = {
            'dc': os.environ.get('DEPLOY_NODE_DC'),
            'level': record.levelname,
            'method': record.funcName,
            'source': record.name,
            'sourceMethod': record.module,
            'host': os.environ.get('DEPLOY_POD_TRANSIENT_FQDN'),
            'file': record.filename,
            'message': record.getMessage(),
            'project': settings.ERROR_BOOSTER_PROJECT,
            'service': os.environ.get('wiki_code'),  # wiki, biz
            'block': os.environ.get('DEPLOY_BOX_ID'),  # backend, console, celery
            'language': 'python',
            'version': get_version(),
            'env': os.environ.get('YENV_TYPE'),  # testing, production, development
            'timestamp': round(record.created * 1000),  # бустер ожидает миллисекунды
            'line': record.lineno,
        }

        self._fill_log_context(log_dict)
        self._fill_exc_info(log_dict, record)

        return log_dict

    @staticmethod
    def _fill_exc_info(log_dict: dict, record: logging.LogRecord):
        if not record.exc_info:
            return

        log_dict['stack'] = ''.join(traceback.format_exception(*record.exc_info))

    @staticmethod
    def _fill_log_context(log_dict: dict):
        log_context_fields = get_log_context()

        if not log_context_fields:
            return

        log_dict['additional'] = log_context_fields
        log_dict['page'] = log_context_fields.get('slug')

        if 'request' in log_context_fields:
            log_dict['reqid'] = log_context_fields['request'].get('id')
            log_dict['url'] = log_context_fields['request'].get('path')

        if 'user' in log_context_fields:
            log_dict['yandexuid'] = log_context_fields['user'].get('id')
            log_dict['ip'] = log_context_fields['user'].get('client_ip')
