import logging

import requests
from django.conf import settings
from django_idm_api.hooks import AuthHooks

from wiki.utils.tvm2 import get_service_ticket

logger = logging.getLogger(__name__)


class GroupNotFoundInIDM(Exception):
    def __init__(self, group_id, group_name, *args, **kwargs):
        super(GroupNotFoundInIDM, self).__init__(args, kwargs)
        self.group_id = group_id
        self.group_name = group_name


class WikiAuthHooks(AuthHooks):
    def _get_or_create_user(self, login):
        # Когда добавляют нового пользователя в группу "Яндекс" или другую
        # группу, которая добавлена в роль "Внутренний", IDM может прислать
        # об этом пуш раньше, чем мы получим этого пользователя через Updatecenter.
        # В этом случае некорректно будет создавать этого пользователя, надо
        # возвращать ошибку из ручки.
        return self._get_user(login)


def get_group_roles(group):
    retries = 0
    while True:
        try:
            service_ticket = get_service_ticket(settings.IDM_TVM2_CLIENT_ID)
            response = requests.get(
                'https://idm-api.yandex-team.ru/api/v1/roles/?'
                'format=json&system={system}&group={group}&type=active&with_parents=true'.format(
                    system=settings.IDM_SYSTEM_NAME, group=group.id
                ),
                headers={'X-Ya-Service-Ticket': service_ticket},
                timeout=10,
            )
            if response.status_code == 400:
                # группа с переданным ID не найдена
                msg = 'Group with id=%d not found in IDM'
                logger.warning(msg, group.id)
                raise GroupNotFoundInIDM(group_id=group.id, group_name=group.name, message=msg, id=group.id)
            elif not response.ok:
                raise Exception('Status %d from IDM API: group.id=%d' % (response.status_code, group.id))
            return [obj['human_short'] for obj in response.json()['objects']]
        except requests.exceptions.Timeout:
            retries += 1
            if retries > 2:
                raise Exception('IDM API retries exceeded')
