import time

from django.core.cache import caches

from wiki.utils.django_redlock.redlock import RedisLock

LEAKYBUCKET_CACHE = caches['leaky_bucket']


class LeakyBucket:
    def __init__(self, key, rate, capacity=None):
        self.key = key
        self.rate = rate
        self.capacity = capacity or rate

    def consume(self, amount=1):
        with RedisLock(self.key, timeout=3, blocking_timeout=3):
            current_amount = self.current_amount()
            required_amount = current_amount + amount
            if required_amount > self.capacity:
                return False

            bucket_data = {
                'amount': required_amount,
                'last_consume': time.time(),
            }
            LEAKYBUCKET_CACHE.set(self.key, bucket_data)
            return True

    def current_amount(self):
        bucket_data = LEAKYBUCKET_CACHE.get(self.key)
        current_amount = 0

        if bucket_data:
            now = time.time()
            elapsed = now - bucket_data['last_consume']
            released = elapsed * self.rate
            current_amount = max(int(bucket_data['amount'] - released) + 1, 0)

        return current_amount
