import functools
import logging
import time

import ylock
from django.conf import settings

logger = logging.getLogger(__name__)

if not isinstance(getattr(settings, 'YLOCK', None), dict):
    raise RuntimeError('Settings for YLOCK must be a dict')

if not isinstance(settings.YLOCK['token'], str):
    raise RuntimeError('YLOCK token must be instance of "str", got "{0}"'.format(type(settings.YLOCK['token'])))

if not isinstance(settings.YLOCK['prefix'], str):
    raise RuntimeError('YLOCK prefix must be instance of "str", got "{0}"'.format(type(settings.YLOCK['prefix'])))


def get_ylock_manager():
    from wiki.utils.ylock_redis.ylock_redis import Manager as RedisManager

    settings_copy = settings.YLOCK_REDIS.copy()
    del settings_copy['backend']
    return RedisManager(**settings_copy)


if settings.USE_MDB_REDIS:
    manager = get_ylock_manager()
else:
    manager = ylock.create_manager(**settings.YLOCK)


def benchmark_ylock_manager(mgr, lock_name):
    # (0.004398391246795654, 0.004379966735839844)
    lock_sum = 0
    unlock_sum = 0
    for i in range(1000):
        lock = mgr.lock(lock_name, block=False, block_timeout=1)
        t_0 = time.time()
        lock.__enter__()
        lock_sum += time.time() - t_0
        t_0 = time.time()
        lock.__exit__(None, None, None)
        unlock_sum += time.time() - t_0
    return lock_sum / 1000, unlock_sum / 1000


class FailedToGetLockError(Exception):
    pass


class LockTaken(FailedToGetLockError):
    pass


def check_lock_acquired(lock_name):
    """
    Проверить занят ли лок, не захватывая его.
    """
    return manager.lock(lock_name).check_acquired()


def execute_with_lock(lock_name, callable, timeout=None):
    """
    Получить лок или бросить исключение FailedToGetLock.

    Если параметр timeout задан, то лок получаем в блокирующем режиме с заданным timeout.

    Если параметр timeout не задан, то лок получаем в неблокирующем режиме.

    @param lock_name Имя лока
    @type lock_name basestring

    @param callable Функция, вызываемая после захвата лока
    @type callable callable

    @param timeout Таймаут блокировки
    @type timeout int
    """

    with manager.lock(lock_name, block=timeout is not None, block_timeout=timeout) as acquired:
        if acquired:
            return callable()
        else:
            raise LockTaken('Lock is taken "{0}"'.format(lock_name))


def get_lock_or_do_nothing(lock_name, timeout=None):
    def wrap(function):
        @functools.wraps(function)
        def do(*args, **kwargs):
            try:
                execute_with_lock(lock_name, functools.partial(function, *args, **kwargs), timeout)
            except FailedToGetLockError:
                logger.info('Did not get lock "%s"', lock_name)

        return do

    return wrap
