import base64
import hashlib
from email.utils import formatdate

import yenv
from django.conf import settings
from django.core.mail import EmailMessage
from django.utils.text import wrap

from wiki.utils import smart_list

EMAIL_BODY_WRAP_SIZE = 78


def build_email_message(to, subject, body, from_email, **params):
    """
    Создать объект django.core.email.EmailMessage со свойствами,
    присущими всем нотификациям в Вики.
    Функция должна быть максимально гибкой и универсальной.
    Для правильного разделения абстракций — ничего не знает про Вики,
    обычно вызывается из build_wiki_email_message
    Все что передается в params — опционально.

    @type to: basestring or iterable
    @type subject:
    @type body:
    @type cc: basestring or iterable
    @type bcc: basestring or iterable
    @type from_email: str
    @param headers: словарь для перезаписи/добавления заголовков к дефолтным
    @type headers: mapping
    @param attachments: прикрепленные файлы
    @param connection: можно передать EmailBackend
    @param content_subtype: 'html' или 'plain' или что там еще.

    @rtype: EmailMessage
    @return: composed message object
    """

    to = smart_list(to)
    cc = smart_list(params.get('cc'))
    bcc = smart_list(params.get('bcc'))
    reply_to = params.get('reply_to')

    headers = {}

    if reply_to is not None:
        headers['Reply-To'] = reply_to

    headers.update(params.get('headers', {}))

    content_subtype = params.get('content_subtype', 'plain')

    if yenv.type != 'production':
        if content_subtype == 'html':
            linebreak = '<br/>\n'
        else:
            linebreak = '\n'

        original_pieces = [
            'Original To: ' + str(to),
            'CC: ' + str(cc),
            'BCC: ' + str(bcc),
        ]
        to = [settings.DEBUG_EMAIL]
        cc = []
        bcc = []

        if not from_email.endswith('yandex-team.ru'):
            # рассылка DEBUG_EMAIL в данный момент закрыта от внешних адресов
            # пошлем письмо с внутреннего адреса
            original_pieces.append('From ' + str(from_email))
            from_email = settings.SERVER_EMAIL

        body_debug_suffix = linebreak * 2
        separator = ',' + linebreak
        body_debug_suffix += separator.join(original_pieces)
        body += body_debug_suffix

    author_name = params.get('author_name', '')

    if '@' in author_name:
        # для случаев когда автор "neofelis@domain.ru neofelis@domain.ru"
        author_name = author_name.split(' ')[0].split('@')[0]

    encoded_from_field = f'{author_name} <{from_email}>'

    message = EmailMessage(
        subject=subject,
        body=wrap(body, EMAIL_BODY_WRAP_SIZE),
        from_email=encoded_from_field,
        to=to,
        cc=cc,
        bcc=bcc,
        attachments=params.get('attachments'),
        headers=headers,
        connection=params.get('connection'),
    )
    message.content_subtype = content_subtype
    return message


def build_wiki_email_message(to, subject, body, **params):
    """
    Обертка над build_email_message, добавляющая вики-специфичные штуки.
    Все параметры, описанные в build_email_message +

    @param supertag: супертег викистрницы, к которой имеет отношение сообщение
    @param author_name: имя автора для подставления в поле From
    @type author_name: basestring
    @param add_references_header: если True — добавить заголовок References
      с указанием страницы, идентифицируемой `supertag`.
    @type add_references_header: bool

    @rtype: EmailMessage
    @return: composed message object
    """
    headers = {
        'Return-Path': settings.SERVER_EMAIL,
        'X-Yandex-Wiki-Env': yenv.type,
    }

    supertag = params.get('supertag')
    if supertag is not None:
        headers['X-Yandex-Wiki-Path'] = supertag

        add_references_header = params.get('add_references_header', True)
        if add_references_header:
            headers['References'] = '<%s.%s@yandex-team.ru>' % (
                hashlib.md5(supertag.encode('utf-8')).hexdigest(),
                settings.WIKI_CODE,
            )

    author_name = params.get('author_name') or 'YandexWiki'
    from_name = ' '.join(['(Wiki)', author_name])
    params['author_name'] = from_name
    params['from_email'] = params.get('from_email', settings.SERVER_EMAIL)

    date_string = formatdate()
    headers['Date'] = date_string
    headers['X-Yandex-Service'] = sign_for_yamail(
        date_string=date_string,
        from_email=from_name + ' ' + settings.SERVER_EMAIL,
        subject=subject,
    )

    if 'headers' in params:
        params['headers'].update(headers)
    else:
        params['headers'] = headers

    return build_email_message(to, subject, body, **params)


def encode_from_for_email_message(from_email, name=''):
    """
    Сформировать забейз64энкоженное имя автора для подставления в поле From имейла.
    НЕ НУЖНО! ДЖАНГА САМА ЭТО УМЕЕТ ДЕЛАТЬ
    @type name: basestring
    @type from_email: str
    @rtype: str
    """
    name_encoded = base64.encodebytes(name.encode('UTF-8'))[:-1].decode('utf-8')
    name_encoded = name_encoded.replace('\n', '')

    return '=?utf-8?B?{name}?= <{email}>'.format(
        name=name_encoded,
        email=from_email,
    )


def sign_for_yamail(date_string, from_email, subject):
    tmp = '_'.join(
        [
            date_string,
            from_email,
            subject,
            settings.MAIL_SIGNATURE_LABEL,
            settings.MAIL_SIGNATURE_SALT,
        ]
    )
    tmp = settings.MAIL_SIGNATURE_LABEL + ' ' + hashlib.md5(tmp.encode('UTF-8')).hexdigest()
    return base64.encodebytes(tmp.encode('utf-8'))[:-1].decode('utf-8')
