from functools import partial
from multiprocessing.dummy import Pool

from django.core.management import BaseCommand
from django.db.models import Q
from django.http import Http404
from django_mds.storage import MDSStorage

from wiki.pages.models import Page, Revision

MDS_STORAGE = MDSStorage()


def get_descendants(supertag, include_page=False):
    qs = Q(supertag__startswith=supertag + '/')
    if include_page:
        qs |= Q(supertag=supertag)
    return Page.objects.filter(qs).order_by('supertag')


def check_content_in_mds(model, values):
    try:
        if not values['mds_storage_id']:
            raise ValueError('mds_storage_id is empty')
        if not MDS_STORAGE.exists(values['mds_storage_id']):
            raise Http404('Object not found in storage')
    except Exception as err:
        print(
            "Can't open {model_name} by link '{link}'. Object ID in database={id}. Error message: '{error}'".format(
                model_name=model.__name__,
                link=MDS_STORAGE.url(values['mds_storage_id']),
                id=values['id'],
                error=err.message,
            )
        )


def check_model_content_by_page_id(model, page_id_list):
    if page_id_list:
        queryset = model.objects.filter(page_id__in=page_id_list)
    else:
        queryset = model.objects.all()
    queryset = queryset.values('id', 'mds_storage_id')

    print('Try to checked {number} objects...'.format(number=len(queryset)))
    Pool(processes=4).map_async(partial(check_content_in_mds, model), queryset).get(99999999)


class Command(BaseCommand):
    help = 'Check content (files, pages, revisions) stored in MDS'

    def add_arguments(self, parser):
        super(Command, self).add_arguments(parser)
        parser.add_argument(
            '--cluster',
            '-c',
            action='store',
            dest='cluster',
            required=False,
            help='Check content in MDS only for some cluster. For example: --cluster "users/blabla"',
        )

    def handle(self, *args, **options):
        from wiki.files.models import File

        cluster = options.get('cluster')

        print(
            'Start checking content of Page, Revision and File models%s...' % " in cluster '%s'" % cluster
            if cluster
            else ''
        )
        print('\nCheck Pages ...')
        if cluster:
            queryset = get_descendants(cluster, False)
        else:
            queryset = Page.objects.all()
        queryset = queryset.values('id', 'mds_storage_id')
        if len(queryset) == 0:
            raise ValueError("Cluster '%s' does not exist" % cluster)
        Pool(processes=4).map_async(partial(check_content_in_mds, Page), queryset).get(99999999)

        if cluster:
            page_id_list = [values['id'] for values in queryset]
        else:
            page_id_list = []

        print('\nCheck Revisions ...')
        check_model_content_by_page_id(Revision, page_id_list)

        print('\nCheck Files ...')
        check_model_content_by_page_id(File, page_id_list)
