
from __future__ import unicode_literals

import json
from random import randint

import six
from django.db import models
from django.forms import ValidationError
from django.utils.encoding import smart_bytes, smart_str
from django.utils.translation import ugettext_lazy as _
from jsonfield.fields import JSONFieldBase


class MdsJsonField(JSONFieldBase, models.CharField):
    """
    Реализует прозрачную для пользователя запись в MDS.

    Инстанс MDS должен быть передан или сконфигурирован в settings.

    @param storage инстанс MDS
    """

    storage = None

    def __init__(self, storage=None, *args, **kwargs):
        self.storage = storage

        kwargs['max_length'] = 250
        super(MdsJsonField, self).__init__(*args, **kwargs)

    def read_from_storage(self, name):
        return smart_str(self.storage.open(name).read())

    def write_to_storage(self, value):
        return self.storage.save(self.make_storage_id(), six.BytesIO(smart_bytes(value)))

    def make_storage_id(self):
        """
        Возвращает строку storage_id для поля.
        """
        return str(randint(10 ** 5, 10 ** 6 - 1))

    def pre_init(self, value, obj):
        """Convert a string value to JSON only if it needs to be deserialized.

        SubfieldBase metaclass has been modified to call this method instead of
        to_python so that we can check the obj state and determine if it needs to be
        deserialized"""

        if obj._state.adding:
            # Make sure the primary key actually exists on the object before
            # checking if it's empty. This is a special case for South datamigrations
            # see: https://github.com/bradjasper/django-jsonfield/issues/52
            if getattr(obj, 'pk', None) is not None:
                if isinstance(value, six.string_types):
                    try:
                        return json.loads(self.read_from_storage(value), **self.load_kwargs)
                    except ValueError:
                        raise ValidationError(_('Enter valid mds storage id'))

        return value

    def get_db_prep_value(self, value, connection, prepared=False):
        value = super(MdsJsonField, self).get_db_prep_value(value, connection=connection, prepared=prepared)
        if isinstance(value, six.string_types):
            return self.write_to_storage(value)
        else:
            return None

    def dumps_for_display(self, value):
        """
        @type value: str
        """
        return value
