
from more_itertools import chunked


def queryset_iterator(queryset, chunk_size=1000, pk_gt=0):
    """
    Возвращает итератор по кверисету, делая в фоне
    несколько запросов и получая объекты кусочками
    размером chunksize.

    Нужен, когда хочется сэкнономить память ценой нескольких запросов в базу.
    Можно использовать, когда нужно обработать большой QuerySet,
    но не хочется загружать его в память (и хранить в кеше
    инстанса QuerySet)

    QuerySet.iterator() должен работать похоже, но, видимо, не работает
    для python mysqldb, который видимо все равно все загружает в память
    перед тем как отдать в Django ORM.

    Особенности:
      * Эта функция пересортирует queryset по pk, так что если есть
      order_by — нужно что-то еще использовать. Сортировка по pk нужна,
      чтобы использовать where pk > xxx вместо offset, который убивает
      производительность, а также для того, чтобы выборка была прдсказуема
      и не изменялась между запросов.
      * В случае, если chunksize кратен len(queryset) в конце будет сделан лишний запрос,
      который получит 0 результатов, потому что изначально неизвестно
      сколько данных селектит кверисет.
      * если во время обработки QuerySet вы создаете в цикле новые объекты, то они
      могут попасть в результат queryset_iterator на очередной итерации. Это может
      быть нежелательно для вас.

    @param pk_gt: Нижняя граница (невключительно) primary key возвращаемых объектов
    @type pk_gt: int
    """
    pk = pk_gt
    queryset = queryset.order_by('pk')

    while True:
        chunk = queryset.filter(pk__gt=pk)[:chunk_size]
        index = 0
        for index, row in enumerate(chunk, start=1):
            pk = row.pk
            yield row

        if index < chunk_size:
            # последний чанк был неполным (или пустым в случае,
            # когда chunk_size кратен количеству объектов)
            return


def get_chunked(queryset, chunk_size=300):
    """
    Generator returns chunks from QuerySet

    @type queryset: QuerySet
    """
    for chunk in chunked(queryset_iterator(queryset, chunk_size), chunk_size):
        yield chunk
