"""
Вспомогательные функции для работы с путями к вики-страницам.

@author: Eugene Voytitsky (voyt@yandex-team.ru)
"""

import regex as re
import urllib.parse

from wiki.users.utils import get_users_cluster_tag


def resolve(curr_path, relative_path):
    """Резолвит относительный путь relative_path в абсолютный.
    Считается что relative_path задан относительно curr_path, который должен быть абсолютным.
    """
    if not relative_path.startswith('/'):
        # ссылка на страницу относительная -> преобразуем к абсолютной
        if relative_path.startswith('!/') or relative_path == '!' or relative_path == '':
            relative_path = curr_path + relative_path[1:]
        elif relative_path == '..':
            relative_path = curr_path + '/..'
        else:
            relative_path = curr_path + '/../' + relative_path
    return relative_path


def transform_relative_path(curr_path, relative_path, octx):
    """Преобразовать относительный путь, заменив в нем вики специфичные символы, такие как '!' """
    if octx.config.is_grid:
        # если это грид, то преобразовываем относительную ссылку в абсолютную, так как при вставке грида
        # с относительными ссылками на другую страницу ссылки ломаются (WIKI-10270)
        return resolve(curr_path, relative_path)

    if relative_path.startswith('!/') or relative_path == '!':
        relative_path = '.' + relative_path[1:]
    elif relative_path.startswith('./'):
        relative_path = '../' + relative_path[2:]
    elif relative_path.startswith('../'):
        relative_path = '../../' + relative_path[3:]
    elif not relative_path.startswith('/') and relative_path != '..':
        # это преобразование необходимо для того, чтобы в дальнейшем из этой относительной ссылки было легко построить
        # абсолютную, например, ссылку на docviewer
        relative_path = '../' + relative_path

    return relative_path


def canonize(path):
    """Канонизирует путь, так что в нём не будет '/.', '/..', '//'."""
    sz = 0
    while sz != len(path):
        sz = len(path)
        path = re.sub(r'/((/+)|(\./)|([^/]+/\.\.(/|$)))|^/\.\.(/|$)', '/', path)
        path = re.sub(r'(?<=.)(/|/\.)$', '', path)

    if path.startswith('/~'):
        # преобразовывание ссылки /~login/page -> /users/login/page
        # или /UserName/page для обратной совместимости со старой системой
        # наименования пользовательских кластеров
        url = path[2:]
        # В случае, если значение login после тильды отсутствует (/~), оставляем ссылку без изменений
        if url == '':
            return path

        # пытаемся получить логин пользователя из url
        pos = url.find('/')
        if pos == -1:
            username = url
            path = ''
        elif pos > 0:
            username = url[:pos]
            path = url[pos:]
        else:
            # В случае, если значение login после тильды отсутствует (/~/page), оставляем ссылку без изменений
            return path

        # Найти пользовательский кластер и вернуть преобразованный путь
        # относительно этого кластера
        users_cluster = get_users_cluster_tag(username)
        if users_cluster:
            return '/' + users_cluster + path

    return path


def get_hostname(url):
    """Парсит url и возвращает hostname."""
    return urllib.parse.urlparse(url).hostname


def is_wiki_url(config, url, ignore_case):
    """Является ли http url ссылкой на вики-страницу?

    Проверяется без учета регистра, если ignore_case=True, иначе регистрозависимо.
    Возвращает:
    1). None, если url – не является вики-ссылкой,
    2). иначе: от начала url отрезается базовый адрес вики (последний слеш не отрезается) и возвращается тапл
      2.1). (url, [url]) - если url не содержит подстроки WIKI_ATTACH_FILE_SUBSTR
      2.2). (url, [page, file]) - если url содержит подстроку WIKI_ATTACH_FILE_SUBSTR и имеет вид:
      url = pageWIKI_ATTACH_FILE_SUBSTRfile
    """
    if ignore_case and config.wiki_base_urls:
        u = url.lower()
    else:
        u = url
    result = None
    for base_url in config.wiki_base_urls:
        if u.startswith(base_url):
            url = url[len(base_url) - 1 :]
            if ignore_case:
                result = url, url.rsplit(config.wiki_attach_file_substr, 1)
                break
            else:
                u = u[len(base_url) - 1 :]
                r = u.rsplit(config.wiki_attach_file_substr, 1)
                if len(r) == 1:
                    result = url, [url]
                    break
                else:
                    result = url, [url[: len(r[0])], url[len(url) - len(r[1]) :]]
                    break
    if result:
        result_1 = result[1]
        if len(result_1) == 2 and result_1[0] == '':
            # файл присоединенный к корневой странице - надо исправить page="" на page="/"
            result_1[0] = '/'
    return result
