import collections


class Peekable(object):
    """
    Iterator with lookahead capability
    """

    def __new__(cls, iterator):
        if isinstance(iterator, cls):
            return iterator
        else:
            return super(Peekable, cls).__new__(cls)

    def __init__(self, iterator):
        if self is iterator:
            return
        self.iterator = iter(iterator)
        self.buffer = collections.deque()

    def __iter__(self):
        return self

    def __next__(self):
        if len(self.buffer) > 0:
            return self.buffer.popleft()
        else:
            return next(self.iterator)

    def peek(self, index=0):
        if self.has_next(index + 1):
            return self.buffer[index]
        else:
            return None

    def peek_many(self, num_items):
        self._load_n_items(num_items)
        return list(self.buffer)[:num_items]

    def has_next(self, num_items=1):
        self._load_n_items(num_items)
        return len(self.buffer) >= num_items

    def _load_n_items(self, num_items):
        while len(self.buffer) < num_items and self._load_next_item():
            pass

    def _load_next_item(self):
        try:
            self.buffer.append(next(self.iterator))
        except StopIteration:
            return False
        else:
            return True

    def takewhile(self, should_take):
        while self.has_next():
            if should_take(self.peek()):
                yield next(self)
            else:
                return

    def undo(self, item):
        self.buffer.appendleft(item)
