"""
Декоратор memoize_for_this_request запоминает на время одного запроса
результат выполнения функции в данном потоке.
"""

import functools
import logging

from django.core.signals import request_finished, request_started

from wiki.utils.injector import thread_locals

logger = logging.getLogger(__name__)


def prepare_for_memoize(*args, **kwargs):
    logger.debug('Cleared memoized values on request started')
    thread_locals._memoize = {}


def clean_after_memoize(*args, **kwargs):
    logger.debug('Cleared memoized values on request finished')
    thread_locals._memoize = {}


def is_memoize_on():
    return hasattr(thread_locals, '_memoize')


def memoize_for_this_request(function):
    """
    Декоратор позволяет для функция с позиционными аргументами
    запоминать их результат на время одного джанго-запроса.

    Например, с его помощью можно получать консистентное состояние
    в рамках одного запроса относительно следующих знаний:
      * сервис только для чтения
      * пользователь - админ
    """

    @functools.wraps(function)
    def wrapper(*args, **kwargs):
        the_hash = '{0}_{1}'.format(function.__name__, hash(args))
        if not hasattr(thread_locals, '_memoize'):
            logger.debug('Not memoizing, not inside django request?')
            return function(*args)
        if the_hash in thread_locals._memoize:
            logger.debug('Remembered value for %s', function)
            return thread_locals._memoize[the_hash]
        result = function(*args)
        thread_locals._memoize[the_hash] = result
        logger.debug('Saved value for %s', function)
        return result

    return wrapper


request_started.connect(prepare_for_memoize)
request_finished.connect(clean_after_memoize)
