import os
from collections import namedtuple
from typing import List

import polib
from django.conf import settings

Domain = namedtuple('domain', ['name', 'must_exist'])
DOMAINS = (
    Domain('django', must_exist=True),
    Domain('djangojs', must_exist=False),
)


class TranslationSelfCheck(Exception):
    pass


def entries_with_missing_translations(po):
    """
    Генератор непереведенных записей в po-файле.

    @type po: POFile
    @rtype: POEntry
    """
    for entry in po:
        if not entry.translated():
            if not entry.occurrences:
                continue
            if not entry.msgstr and entry.msgstr_plural:
                # У некоторых записей есть msgstr_plural, но нет msgstr.
                # Их считаем переведенными, если они не obsolete и не fuzzy!
                if not entry.obsolete and 'fuzzy' not in entry.flags:
                    continue
            yield entry


def missing_keys(po):
    """
    Генератор непереведенных ключей.

    @type po: POFile
    @rtype: str
    """
    for entry in entries_with_missing_translations(po):
        if not entry.msgstr:
            yield entry.msgid
        else:
            yield entry.msgid_plural


def check_files(filenames, skip_keys=None):
    """
    Генератор отсутствующих ключей из po-файлов.

    @type filenames: iterable
    @param skip_keys: Пропускать заданные ключи.
    @type skip_keys: list
    @rtype: str
    """
    if skip_keys is None:
        skip_keys = tuple()

    for filename in filenames:
        for key in missing_keys(polib.pofile(filename)):
            if key in skip_keys:
                continue
            raise TranslationSelfCheck(f'The key is not fully translated: "{key}"')


def _generate_filenames(verbosity: int) -> List[str]:
    filenames = []

    for domain in DOMAINS:
        localedir = os.path.abspath('locale')
        languages = [lang[0] for lang in settings.LANGUAGES]
        for locale in languages:
            basedir = os.path.join(localedir, locale, 'LC_MESSAGES')
            pofile = os.path.join(basedir, '%s.po' % domain.name)

            if verbosity > 0:
                print('processing file "{0}"'.format(pofile))

            if domain.must_exist and not os.path.exists(pofile):
                raise TranslationSelfCheck(f'pofile is obligatory and does not exist: "{pofile}"')

            if os.path.exists(pofile):
                filenames.append(pofile)

    return filenames


def check_translation_integrity(verbosity=2, skip_keys=None):
    check_files(_generate_filenames(verbosity), skip_keys or tuple())
