from django.core.exceptions import ObjectDoesNotExist


class Registry(object):
    """Read description of basic usage
        0) create registry
        1) Register entities you want to get a lot of times
        2) Call get and all of them go into Registry in a single call

    Use it if you want to have a lot of objects of same django Model, but do not know, when

        >>> registry = Registry(Revision.objects, "pk")
        >>> registry.want_to_have(2)
        ...
        >>> registry.want_to_have(150)
        ...
        >>> registry.get(150)
        Revision
        >>> registry.get(2)
        raised ObjectDoesNotExist
        >>> registry.get(1)
        None
    """

    registry = None
    qs = None
    field_name = None

    queried = False

    def __init__(self, qs, field_name):
        self.registry = {}
        self.qs = qs
        self.field_name = field_name

    def want_to_have(self, value):
        self.registry[value] = None

    def get(self, value):
        if value not in self.registry:
            return None
        if not self.queried:
            query_set = self.qs.filter(**{self.field_name + '__in': list(self.registry.keys())})
            for entry in query_set:
                self.registry[getattr(entry, self.field_name)] = entry
            self.queried = True
        if self.registry[value] is None:
            raise ObjectDoesNotExist()
        return self.registry[value]
