import re
import uuid
from typing import List, Optional, Dict, Any
from urllib.parse import parse_qs

from django_yauth.util import get_real_ip
from django.http import HttpRequest

REGEX_ROUTE_PARAM = re.compile(r'\(\?P<(.*)>.*\)\??')


def extract_normalized_route(request: HttpRequest) -> Optional[str]:
    try:
        if request.resolver_match:
            if hasattr(request.resolver_match, 'route'):
                return re.sub(REGEX_ROUTE_PARAM, '<\\1>', request.resolver_match.route)
            else:
                return request.resolver_match.func.__name__
    except Exception:  # noqa
        pass

    return None


def build_request(request: HttpRequest, headers: List[str]) -> Dict[str, Any]:
    return {
        'id': _get_request_id(request),
        'method': request.method,
        'path': request.get_full_path(),
        'headers': dict(_get_headers_values(request, headers)),
        'query_params': _get_query_params(request),
    }


def _get_request_id(request: HttpRequest) -> str:
    return (
        request.META.get('HTTP_X_REQ_ID', None)
        or request.META.get('HTTP_X_REQUEST_ID', None)
        or 'auto-' + uuid.uuid4().hex
    )


def _get_headers_values(request: HttpRequest, headers: List[str]):
    for header in headers:
        value = request.META.get(header)
        if value is not None:
            yield header, value


def _get_query_params(request: HttpRequest) -> Dict[str, str]:
    return parse_qs(request.META.get('QUERY_STRING') or request.GET.urlencode() or '')


def build_user(request: HttpRequest) -> Optional[Dict[str, str]]:
    base = {'client_ip': get_real_ip(request)}

    if hasattr(request, 'user') and request.user and request.user.is_authenticated:
        base.update(
            {
                'id': request.user.id,
                'login': request.user.username,
            }
        )

    return base


def build_org(request: HttpRequest) -> Optional[Dict[str, str]]:
    if hasattr(request, 'org'):
        return request.org and {
            'dir_id': request.org.dir_id,
            'cloud_id': request.org.cloud_id,
        }


def build_auth(request: HttpRequest) -> Optional[Dict[str, Any]]:
    if getattr(request, 'yauser', None) is None:
        return {'mechanism': None}
    if not request.yauser.is_authenticated():
        return {'mechanism': None}
    if not request.yauser.authenticated_by:
        return {'mechanism': 'unknown - request.yauser.authenticated_by is None'}

    ctx = {'mechanism': request.yauser.authenticated_by.mechanism_name}
    if request.yauser.authenticated_by.mechanism_name == 'oauth':
        ctx['application'] = request.yauser.blackbox_result.oauth.client_name
    if request.yauser.authenticated_by.mechanism_name == 'tvm':
        ctx['application'] = request.yauser.service_ticket.src
    return ctx


def build_slug(request: HttpRequest) -> Optional[str]:
    if hasattr(request, 'supertag'):
        return request.supertag
    elif hasattr(request, 'page'):
        return request.page.slug
