import logging
import time

from django.conf import settings
from django.http import HttpRequest, HttpResponse
from ylog.context import log_context

from wiki.utils.request_logging.context import (
    build_auth,
    build_org,
    build_request,
    build_slug,
    build_user,
    extract_normalized_route,
)

logger = logging.getLogger('request_logger')

SLOW_WARM_MS = getattr(settings, 'REQUEST_LOGGING_SLOW_WARN_MS', 40_000)
SLOW_ERR_MS = getattr(settings, 'REQUEST_LOGGING_SLOW_ERR_MS', 120_000)

IGNORED_ENDPOINTS = frozenset(['/_api/frontend/.is_readonly', '/_api/dir/changed', '/ping_unistat', '/unistat'])
LOGGING_HEADERS = [
    'HTTP_X_CLOUD_UID',
    'HTTP_X_UID',
    'HTTP_X_ORG_ID',
]


class RequestLoggingMiddleware:
    """
    Логирование всех запросов
    """

    def __init__(self, get_response=None):
        self.get_response = get_response

    def __call__(self, request: HttpRequest) -> HttpResponse:
        request.start_time = start_time = time.time()

        try:
            request_ctx = build_request(request, headers=LOGGING_HEADERS)
        except Exception:  # noqa
            logger.exception('Cant gather request details')
            request_ctx = None

        with log_context(request=request_ctx):
            response = self.get_response(request)
            execution_time_ms = int((time.time() - start_time) * 1000)

            try:
                with log_context(
                    execution_time_ms=execution_time_ms,
                    status_code=response.status_code,
                    route=extract_normalized_route(request),
                    user=build_user(request),
                    org=build_org(request),
                    auth=build_auth(request),
                    slug=build_slug(request),
                ):
                    self._log_response(request, response, execution_time_ms)
            except Exception:  # noqa
                logger.exception('Request logging failed')

        return response

    @staticmethod
    def _log_response(request: HttpRequest, response: HttpResponse, execution_time_ms: int):
        log_level = logging.DEBUG
        parts = [f'[{response.status_code}] {request.method} {request.get_full_path()}']

        if response.status_code in {400, 409, 500, 503} and hasattr(response, 'data'):
            if error_code := response.data.get('error', {}).get('error_code'):
                parts.append(f'error_code: {error_code}')

        if execution_time_ms > SLOW_ERR_MS:
            log_level = logging.ERROR
            parts.append('close to timeout')

        elif execution_time_ms > SLOW_WARM_MS:
            log_level = logging.WARN
            parts.append('slow')

        if response.status_code >= 500:
            log_level = logging.ERROR

        # Некоторые вьюшки спамятся фронтом и захламляют логи. Но если что-то ломается - логгируем
        if request.get_full_path() in IGNORED_ENDPOINTS and log_level == logging.DEBUG:
            return

        logger.log(log_level, '; '.join(parts))


class LoggingUserOrgMiddleware:
    """
    Логирование пользователя и организации
    """

    def __init__(self, get_response=None):
        self.get_response = get_response

    def __call__(self, request: HttpRequest) -> HttpResponse:
        try:
            user = build_user(request)
            org = build_org(request)
            auth = build_auth(request)
        except Exception:  # noqa
            user = org = auth = None
            logger.exception('LoggingUserOrg failed')

        with log_context(user=user, org=org, auth=auth):
            return self.get_response(request)
