
from django.db.models import ObjectDoesNotExist
from django.utils.translation import ugettext_lazy as _
from rest_framework import serializers

from wiki.pages.dao import page as page_dao
from wiki.pages.logic import tags
from wiki.utils.errors import ValidationError as WikiValidationError


class ModelChoiceField(serializers.Field):

    default_error_messages = {
        'does_not_exist': _("Object with pk={pk} doesn't exist"),
    }

    def __init__(self, *args, **kwargs):
        self.queryset = kwargs.pop('queryset')
        super(ModelChoiceField, self).__init__(*args, **kwargs)

    def to_representation(self, value):
        return value and value.pk

    def to_internal_value(self, data):
        pk = data

        try:
            return self.queryset.get(pk=pk)
        except ObjectDoesNotExist:
            self.fail('does_not_exist', pk=pk)


class TagField(serializers.CharField):
    """
    Используется в сериализаторах, ожидающих тег в одном из параметров.
    Диапазон длины тега фиксирован.
    Проверяем валидность тега

    @param page_should_exist: должна ли существовать страница
    @type page_should_exist: boolean or None, если None не проверяем
    """

    default_error_messages = {
        # Translators:
        #  ru: Ожидалось, что страница `{tag}` существует
        #  en: It is expected, that page `{tag}` exists
        'page_should_exist': _('It is expected, that page `{tag}` exists'),
        # Translators:
        #  ru: Ожидалось, что страница `{tag}` не существует
        #  en: It is expected, that page `{tag}` doesn't exist
        'page_shouldnt_exist': _("It is expected, that page `{tag}` doesn't exist"),
        # Translators:
        #  ru: Строка "{string}" не может быть тегом страницы
        #  en: String "{string}" cannot be a page tag
        'cannot_be_a_tag': _('String "{string}" cannot be a page tag'),
    }

    def __init__(self, **kwargs):
        self.page_should_exist = kwargs.pop('page_should_exist', None)
        super(TagField, self).__init__(allow_blank=False, min_length=1, max_length=255, **kwargs)

    def to_internal_value(self, data):
        tag = self.clean_tag(value=data)
        self.check_page_existence(tag=tag)
        return tag

    def clean_tag(self, value):
        try:
            return tags.string_to_tag(string=value)
        except WikiValidationError:
            self.fail('cannot_be_a_tag', string=value)

    def check_page_existence(self, tag):
        if self.page_should_exist is None:
            return

        page_exists = page_dao.page_exists(tag)

        if page_exists and not self.page_should_exist:
            self.fail('page_shouldnt_exist', tag=tag)
        elif not page_exists and self.page_should_exist:
            self.fail('page_should_exist', tag=tag)
