from django.core.files.storage import Storage
from django.utils.deconstruct import deconstructible

from wiki.api_v2.public.upload_sessions.exceptions import StorageUploadError
from wiki.uploads.s3_client import S3_CLIENT
from wiki.utils.features.get_features import get_wiki_features
from wiki.utils.mock_storage import InMemoryStorage
from wiki.utils.s3.exceptions import S3StorageError


@deconstructible
class S3Storage(Storage):
    def exists(self, name):
        try:
            return S3_CLIENT.key_exists(key=name)
        except StorageUploadError:
            raise S3StorageError()

    def delete(self, name):
        try:
            return S3_CLIENT.delete_objects(keys=[name])
        except StorageUploadError:
            raise S3StorageError()

    def _open(self, name, mode='rb'):
        try:
            return S3_CLIENT.get_object_body_stream(key=name)
        except StorageUploadError:
            raise S3StorageError()

    def save(self, name, content, max_length=None):
        try:
            content.seek(0)
            S3_CLIENT.put_object(key=name, body=content)
            return name
        except StorageUploadError:
            raise S3StorageError()


if get_wiki_features().inmemory_s3storage:
    S3_STORAGE = InMemoryStorage()
else:
    S3_STORAGE = S3Storage()
