"""
    This module provides helper functions related to page subscription process.
"""

from datetime import timedelta

from wiki.utils import timezone
from wiki.notifications.models import PageEvent


def generate_event(event_type, author, page, cluster=None, other_subscribed_user=None, comment=None):
    """
    Create subscription event in database of type 'watch' (or 'unwatch') when
    the django user subscribes to (or unsubscribes from) page or cluster.

    If the event happens for a single page (via /.watch or /.unwatch),
    this page object is passed as `page` and `cluster` is None.

    If the event happens for a cluster (via /.masswatch or /.massunwatch),
    the root page of the cluster is passed as `page` and `cluster` is a list
    of ALL pages the user actually subscribed to, `cluster` may or may not
    include root supertag.
    """
    assert event_type in ['watch', 'unwatch', 'subscribe_other_user']
    event = PageEvent(
        author=author,
        page=page,
        timeout=timezone.now() + timedelta(minutes=20),
        event_type=getattr(PageEvent.EVENT_TYPES, event_type),
    )
    if cluster is not None:
        meta = {'is_cluster': True, 'pages': cluster}
        if comment:
            meta['comment'] = comment
        if other_subscribed_user:
            meta['subscribed_user_id'] = other_subscribed_user.id
        event.meta = meta
    else:
        event.meta = {}
    event.save()
    return event
