"""
Асинхронный таск для ведения ссылок между страницами
"""

from celery.utils.log import get_task_logger

from wiki.org import org_ctx
from wiki.utils.wfaas.errors import WfaasAccessDenied, WfaasApiRequestError
from wiki.utils.tasks.base import CallableTask


class TrackLinksTask(CallableTask):
    name = 'wiki.track_links'
    logger = get_task_logger(__name__)
    time_limit = 60  # 60 сек.

    def run(self, page_id, is_grid, is_new_page, *args, **kwargs):
        # Импорты тут, чтобы избавиться от циклических импортов.
        #  Проблемы возникают с Grid и track_links.
        from wiki.pages.models import Page
        from wiki.grids.models import Grid
        from wiki.pages.logic.backlinks import track_links

        page = Grid.objects.get(id=page_id) if is_grid else Page.objects.get(id=page_id)

        if page.page_type not in [Page.TYPES.PAGE, Page.TYPES.GRID]:
            return

        if page.page_type == Page.TYPES.GRID and not is_grid:
            # ошибочка вышла (такое бывает в b2b)
            page = Grid.objects.get(id=page_id)

        self.logger.info(f'Tracking links for #{page_id}, is_grid: {is_grid}, new_page: {is_new_page}')

        with org_ctx(page.org):
            try:
                track_links(page, is_new_page)
            except WfaasAccessDenied:
                self.logger.error(f'Tracking links for #{page_id} failed: access denied')
            except WfaasApiRequestError:
                self.logger.error(f'Tracking links for #{page_id} failed: api request error')
            finally:
                self.logger.info(f'Tracking links for #{page_id}, is_grid: {is_grid}, new_page: {is_new_page} is done')
