"""
    This module provides {% render %} tag, that stores rendered content
    to a var.
    Example:
        {% load render %}
        {% render to=MYVAR %}
            Some content
        {% endrender %}
        {{ MYVAR }}{{ MYVAR }}
    Will produce
        Some contentSome content
"""

from django import template
from django.utils.safestring import mark_safe

register = template.Library()


@register.tag
def render(parser, token):
    tag_contents = token.split_contents()
    if len(tag_contents) < 2 or not tag_contents[1].startswith('to='):
        raise template.TemplateSyntaxError('"render" tag requires argument to=VAR')
    tovar_name = tag_contents[1][len('to=') :]
    content_nodes = parser.parse(('endrender',))
    parser.next_token()
    return RenderNode(tovar_name, content_nodes)


class RenderNode(template.Node):
    def __init__(self, tovar_name, content_nodes):
        self.tovar_name = tovar_name
        self.content_nodes = content_nodes

    def render(self, context):
        content = self.content_nodes.render(context).strip()
        context[self.tovar_name] = mark_safe(content)
        return ''
