"""
    This module provides {% surround %} tag, that wraps content depending
    on value of flag.

    Example:

        {% load surround %}
        {% surround if=MYFLAG %}
            <div class="boo">
        {% content %}
                Some content here
        {% endcontent %}
            </div>
        {% endsurround %}

    If variable MYFLAG is True, this template will produce

        <div class="boo">
            Some content here
        </div>

    Otherwise it will produce only

        Some content here


    Example above is equivalent to

        {% if MYFLAG %}
            <div class="boo">
        {% endif %}
                Some content here...
        {% if MYFLAG %}
            </div>
        {% endif %}

    Use {% surround %} if you don't like to duplicate {% if %}'s.


    If optional "spaceless" argument is given, the spaces between content
    and its surrounding will be removed.

    Example with "spaceless":

        {% load surround %}
        {% surround if=SOME_TRUE_VALUE spaceless %}
            <div class="boo">
        {% content %}
                Some content here...
        {% endcontent %}
            </div>
        {% endsurround %}

    Produces

        <div class="boo">Some content here</div>
"""

from django import template

register = template.Library()


@register.tag
def surround(parser, surround_token):
    tag_contents = surround_token.split_contents()

    # Save name of var, that will tell us whether we should render surrounding

    if len(tag_contents) < 2 or not tag_contents[1].startswith('if='):
        raise template.TemplateSyntaxError('"surround" tag requires argument if=VAR')
    ifvar_name = tag_contents[1][len('if=') :]

    # Deside whether content should be rendered without spaces around it

    if len(tag_contents) > 2:
        if tag_contents[2] != 'spaceless':
            raise template.TemplateSyntaxError(
                'second argument to "surround" tag may be only word ' '"spaceless" (without quotes)'
            )
        spaceless = True
    else:
        spaceless = False

    # Get template content between tags

    beginning_nodes = parser.parse(('content',))
    parser.next_token()
    content_nodes = parser.parse(('endcontent',))
    parser.next_token()
    ending_nodes = parser.parse(('endsurround',))
    parser.next_token()

    # Initialize Node

    return SurroundNode(ifvar_name, spaceless, beginning_nodes, content_nodes, ending_nodes)


class SurroundNode(template.Node):
    def __init__(self, ifvar_name, spaceless, beginning_nodes, content_nodes, ending_nodes):
        self.ifvar = template.Variable(ifvar_name)
        self.spaceless = spaceless
        self.beginning_nodes = beginning_nodes
        self.content_nodes = content_nodes
        self.ending_nodes = ending_nodes

    def render(self, context):

        # Determine whether we need surrounding

        try:
            should_surround = self.ifvar.resolve(context)
        except template.VariableDoesNotExist:
            should_surround = False

        # Render beginning

        if should_surround:
            beginning = self.beginning_nodes.render(context)
            if self.spaceless:
                beginning = beginning.rstrip()
        else:
            beginning = ''

        # Render content

        content = self.content_nodes.render(context)
        if self.spaceless:
            content = content.strip()

        # Render ending

        if should_surround:
            ending = self.ending_nodes.render(context)
            if self.spaceless:
                ending = ending.lstrip()
        else:
            ending = ''

        # Return rendered result

        return beginning + content + ending
