"""
    This provides {% walktree %} tag.

    {% walktree %} traverses every node of a python structure that represents
    a tree, and applies its template to it.

    Syntax:

        {% walktree DATA %}
            START SUBTREE TEMPLATE
        {% node %}
            NODE TEMPLATE
        {% endnode %}
            END SUBTREE TEMPLATE
        {% endwalktree %}

    DATA is a context variable holding tree structure. START BRANCH TEMPLATE,
    END BRANCH TEMPLATE - templates that apply before and after rendering
    a subtree. NODE TEMPLATE - template that applies to every node.

    Python structure of a tree should be a flat list of objects:

        [
            object1,
            object2,
            object3,
            ...
        ]

    where every object has attribute "level" (or key "level" for dicts).

    Example:

    Structure:

        [
            {'value': 'A', 'level': 5},
            {'value': 'B', 'level': 5},
            {'value': 'C', 'level': 6},
            {'value': 'D', 'level': 7},
            {'value': 'E', 'level': 5},
        ]

    and template:

        {% load walktree %}
        {% walktree t %}
            <ul>
        {% node %}
            <li>Hello, {{ walktree.node.value }}! {{ walktree.subtree }}</li>
        {% endnode %}
            </ul>
        {% endwalktree %}

    The output will be:

        <ul>
            <li>Hello, A!</li>
            <li>Hello, B!
                <ul>
                    <li>Hello, C!
                        <ul>
                            <li>Hello, D!</li>
                        </ul>
                    </li>
                </ul>
            </li>
            <li>Hello, E!</li>
        </ul>

    Special context variables:

    The templates inside {% walktree %} have access to the following context
    variables:

        walktree.node       Current node, e.g {'value': 'B', 'level': 5}
        walktree.subtree    Rendered subtree for current node, or '' if current
                            node doesn't have subnodes
        walktree.level      Level of the current node

    ATTENTION: {% walktree %} doesn't automatically include rendered subtree
    of the node into its output. You should explicitly use
    {{ walktree.subtree }} in the node template.

    If optional parameter "nowrap" is given to {% walktree %} tag, the outmost
    SUBTREE template won't be applied.

    In the example above:

        {% walktree t nowrap %}
            ...

    will produce the same output, but without the outermost <ul> tag.

"""

from django import template
from django.utils.safestring import mark_safe

register = template.Library()


@register.tag
def walktree(parser, token):
    contents = token.split_contents()
    if 'nowrap' in contents:
        should_wrap = False
        contents.remove('nowrap')
    else:
        should_wrap = True
    try:
        tag_name, data_varname = contents
    except ValueError:
        raise template.TemplateSyntaxError('walktree tag requires an argument -- tree data')
    branch_start = parser.parse(('node',))
    parser.delete_first_token()
    node = parser.parse(('endnode',))
    parser.delete_first_token()
    branch_end = parser.parse(('endwalktree',))
    parser.delete_first_token()
    return WalktreeNode(
        data_varname,
        should_wrap,
        branch_start,
        node,
        branch_end,
    )


class WalktreeNode(template.Node):

    level_template = template.Template('{{ node.level }}')

    def __init__(self, data_varname, should_wrap, branch_start, node, branch_end):
        self.data_var = template.Variable(data_varname)
        self.should_wrap = should_wrap
        self.branch_start = branch_start
        self.node = node
        self.branch_end = branch_end

    def render(self, context):
        try:
            nodes = self.data_var.resolve(context)
        except template.VariableDoesNotExist:
            return ''
        if not isinstance(nodes, (list, tuple)):
            return ''
        context.push()
        output = mark_safe(self.render_subtree(nodes, context, should_wrap=self.should_wrap))
        context.pop()
        return output

    def render_subtree(self, nodes, context, should_wrap=True):
        output = ''
        if len(nodes) == 0:
            return ''
        current_level = self.level(nodes[0])
        context['walktree'] = {
            'level': current_level,
        }
        if should_wrap:
            output += self.branch_start.render(context)
        context.push()
        for node, subtree in self.walk_level(nodes, current_level):
            context.push()
            rendered_subtree = self.render_subtree(subtree, context)
            context.pop()
            context['walktree'] = {
                'node': node,
                'subtree': mark_safe(rendered_subtree),
                'level': current_level,
            }
            output += self.node.render(context)
        context.pop()
        if should_wrap:
            output += self.branch_end.render(context)
        return output

    def walk_level(self, nodes, level):
        while True:
            if len(nodes) == 0:
                return
            node_level = self.level(nodes[0])
            if node_level < level:
                return
            if node_level > level:  # starnge!?
                node = None
            else:
                node = nodes[0]
                nodes = nodes[1:]
            subtree = []
            nodes_it = iter(nodes)
            while True:
                try:
                    next_node = next(nodes_it)
                except StopIteration:
                    break
                next_node_level = self.level(next_node)
                if next_node_level <= node_level:
                    break
                subtree.append(next_node)
            yield node, subtree
            nodes = nodes[len(subtree) :]

    def level(self, node):
        level_context = template.Context({'node': node})
        level_str = self.level_template.render(level_context)
        try:
            return int(level_str)
        except ValueError:
            return 0
