
import logging
from datetime import datetime

import pytz
from django.utils import timezone as original_timezone

logger = logging.getLogger(__name__)

MSK_TZ = pytz.timezone('Europe/Moscow')


def make_aware(timezone, *args):
    """
    Сделать дату timezone-aware.

    В args принимает либо объект datetime, либо аргументы для конструктора datetime

    Идентичные вызовы:
    > make_aware('utc', datetime(2014, 01, 20))
    > make_aware('utc', 2014, 01, 20)

    > make_aware('current', datetime(2014, 01, 20))

    @type timezone: tzinfo
    @param timezone: либо 'utc', либо 'current'
    @rtype: datetime
    """
    assert len(args) > 0
    dt = None
    if len(args) == 1:  # либо первый аргумент - datetime, либо год
        if isinstance(args[0], datetime):
            dt = args[0]  # остальные аргументы игнорируются
    if dt is None:
        dt = datetime(*args)

    return original_timezone.make_aware(dt, timezone)


def make_aware_utc(*args):
    """
    Сделать наивную дату timezone-aware в UTC
    """
    return make_aware(original_timezone.utc, *args)


def make_aware_current(*args):
    """
    Сделать наивную дату timezone-aware в текущей таймзоне.
    """
    return make_aware(original_timezone.get_current_timezone(), *args)


def now():
    """
    Просто алиас, чтобы избежать ненужных импортов.
    """
    return original_timezone.now()


def now_local():
    """
    Текущее время в таймзоне пользователя
    """
    return make_naive_current(now())


def select_timezone(user, default_tz='Europe/Moscow'):
    if not user.is_authenticated:
        # для неаутентифицированных пользователей вернем зону по-умолчанию
        logger.debug('User is not authenticated, chose default timezone %s', default_tz)
        return default_tz
    profile = user.staff
    if profile.tz:
        return profile.tz
    else:
        logger.warn('User does not have tz set, using default "%s" uid="%s"', default_tz, profile.uid)
        return default_tz


def validate_request_time_str(time_str):
    """
    Validate datetime stamp from django request and return datetime object
    @param time_str: datetime as string
    @raise: ValueError
    @rtype: datetime
    """
    if time_str:
        return make_aware_utc(datetime.strptime(time_str, '%Y-%m-%d %H:%M:%S'))


def make_naive_utc(dt):
    return original_timezone.make_naive(dt, original_timezone.utc)


def make_naive_current(dt):
    return original_timezone.make_naive(dt, original_timezone.get_current_timezone())


def make_naive_msk(dt):
    """
    Вернуть дату, наивную в часовом поясе МСК.
    """
    return original_timezone.make_naive(dt, MSK_TZ)
