import logging
import re
import requests

from ids import exceptions
from django.conf import settings
from .shortcuts import get_tracker_issues_repository
from .utils import get_users_logins_by_uids

logger = logging.getLogger(__name__)

TICKET_FORMAT_REGEX = re.compile(r'([A-Z]{2,})-(\d+)')


def get_startrek_issues(issue_keys, user_auth, issue_processor):
    """
    @param issue_keys: итератор с ключами тикетов (строки вида "ABC-1000")
    @param issue_processor: наследник IssueProcessor, который содержит
      список нужных полей и метод для преобразования полей для
      использования в вики
    @return словарь <ключ тикета (строка), объект с данными тикета
    (результат build_..._ticket_func)>
    """
    if not issue_keys:
        return {}

    try:
        issues_collection = get_tracker_issues_repository(user_auth=user_auth)

        issue_keys = [key.upper().strip() for key in set(issue_keys)]
        issue_keys = [x for x in issue_keys if TICKET_FORMAT_REGEX.match(x) is not None]

        BATCH_SIZE = 100
        found_issues = []

        for st_idx in range(0, len(issue_keys), BATCH_SIZE):
            issue_keys_batch = issue_keys[st_idx : st_idx + BATCH_SIZE]
            found_issues_batch = issues_collection.get_all(
                keys=','.join(issue_keys_batch),
                expand='aliases',
                localized='false',
                retries=5,
                timeout=2,
                fields=issue_processor.expected_st_fields,
            )
            found_issues += found_issues_batch

    except exceptions.IDSException as exc:
        log_level = logging.ERROR

        if isinstance(exc, exceptions.BackendError) and exc.status_code == 401:  # Пользователь не авторизован в трекере
            log_level = logging.WARNING

        logger.log(log_level, 'IDS error: %s', repr(exc))
        return {}

    except requests.exceptions.ReadTimeout as exc:
        logger.warning('Tracker timeout: %s', repr(exc))
        return {}

    issues = {}

    if found_issues and settings.IS_BUSINESS:
        # В found_issue в атрибуте id у пользователя в b2b находится uid, а в Интранете – логин, поэтому необходимо
        # заменить все uid на login в объектах 'assignee' и 'createdBy'
        uids = [issue_processor.safe_dict_key(issue, 'assignee', 'id') for issue in found_issues if issue['assignee']]
        uids.extend(
            [issue_processor.safe_dict_key(issue, 'createdBy', 'id') for issue in found_issues if issue['createdBy']]
        )
        uid_2_login_dict = get_users_logins_by_uids(uids)
        for found_issue in found_issues:
            if found_issue['assignee']:
                login = uid_2_login_dict.get(issue_processor.safe_dict_key(found_issue, 'assignee', 'id'))
                found_issue['assignee'].id = login or issue_processor.display(found_issue['assignee'], 'en')
            if found_issue['createdBy']:
                login = uid_2_login_dict.get(issue_processor.safe_dict_key(found_issue, 'createdBy', 'id'))
                found_issue['createdBy'].id = login or issue_processor.display(found_issue['createdBy'], 'en')

    for key in issue_keys:
        for startrek_issue in found_issues:
            issue_key = startrek_issue['key']
            aliases = startrek_issue['aliases']

            if issue_key == key or key in aliases:
                wrapped_issue = issue_processor.process(startrek_issue)
                if wrapped_issue is not None:
                    issues[key] = wrapped_issue
                    break
    return issues
