import logging

from django.conf import settings

from wiki.utils.blackbox import SERVICE_IP
from wiki.sync.connect.org_ctx import get_org_dir_id
from yandex_tracker_client import TrackerClient
from wiki.utils.tracker.ids_st.connector import TrackerConnection
from wiki.utils.tvm2 import get_user_ticket, get_service_ticket


logger = logging.getLogger('trackerClient')
STARTREK_READ_TIMEOUT = 5


def _make_tvm2_user_ticket():
    return get_user_ticket(
        user_ip=SERVICE_IP,
        server_host=settings.API_WIKI_HOST,
        oauth_token=settings.STARTREK_CONFIG['OAUTH2_TOKEN'],
        session_id=None,
    )


def get_tracker_connection(user_auth=None, as_service=False) -> TrackerConnection:
    kwargs = {}
    if settings.IS_BUSINESS:
        kwargs['org_id'] = get_org_dir_id()

    if as_service:
        kwargs['api_version'] = 'service'  # queues, etc;
    elif user_auth and user_auth.cloud_uid:  # облачная авторизация ... но
        if user_auth.uid:
            # в трекере (и коннекте) старый паспортный пользователь может не получить cloud_uid - и мы получим
            # 401 not found :) поэтому если знаем uid - используем его
            kwargs['uid'] = user_auth.uid
        else:
            kwargs['cloud_uid'] = user_auth.cloud_uid
        logger.info(f'Kwargs to tracker {kwargs}')
    else:
        kwargs['user_ticket'] = _get_user_ticket_from_auth(user_auth)

    return TrackerConnection(
        user_agent='wiki',
        timeout=STARTREK_READ_TIMEOUT,
        retries=3,
        host=settings.TRACKER_HOST,
        service_ticket=get_service_ticket(settings.STARTREK_TVM2_CLIENT_ID),
        **kwargs,
    )


def get_tracker_client(user_auth, as_service=False):
    http_connector = get_tracker_connection(user_auth, as_service)
    return TrackerClient(connection=http_connector)


def _get_user_ticket_from_auth(user_auth):
    if user_auth:
        # Нам передали объект UserAuth
        if user_auth.tvm2_user_ticket is not None:
            # к нам пришел сервис с авторизацией по tvm2 вместе с tvm2_user_ticket которого он представляет
            user_ticket = user_auth.tvm2_user_ticket
        elif user_auth.tvm2_user_ticket is None and user_auth.is_tvm_authentication:
            # к нам пришел сервис с авторизацией по tvm2 но без пользователя. К примеру, это поисковый бот
            user_ticket = _make_tvm2_user_ticket()
        else:
            # user_auth.oauth_token создается через механизм get_token_by_sessionid, который
            # использует user_auth.sessionid, либо берет из кеша по user_auth.uid
            # если user_auth.sessionid = None, то tvm2 вернет ошибку "Cookie header is missing"
            # здесь в теории еще могут быть какие-то проблемы, но кейс с поисковым
            # роботом по tvm2 (WIKI-13584) будет разрешен выше

            # @todo change this to exception after 25 of april 2020
            logger.warn('Deprecated auth method was used by %s' % (user_auth.server_host))
            user_ticket = get_user_ticket(
                user_auth.user_ip,
                user_auth.server_host,
                user_auth.get_oauth_token() if settings.IS_INTRANET else None,
                user_auth.sessionid,
            )
    else:
        user_ticket = _make_tvm2_user_ticket()
    return user_ticket
