from isodate import isoduration


class IssueProcessor(object):
    """
    Базовый класс для наследования и передачи в get_startrek_issues.
    Наследники должны описывать список необходимых стартрек-полей
    в expected_st_fields и имплементить process для преобразования
    тикета из ids в формат нужный в вики.
    """

    expected_st_fields = None

    @classmethod
    def process(cls, issue_data):
        raise NotImplementedError

    @staticmethod
    def safe_dict_key(obj, key, attr):
        try:
            return getattr(obj[key], attr)
        except (KeyError, TypeError, AttributeError):
            return

    @staticmethod
    def process_user(issue_data, key):
        return IssueProcessor.safe_dict_key(issue_data, key, 'id')

    @staticmethod
    def display(data, lang):
        if data is None:
            return ''

        if isinstance(data.display, dict):
            return data.display.get(lang, data.display['en'])
        else:
            return data.display

    @staticmethod
    def process_dateteime(value):
        from dateutil.parser import parse
        import pytz

        return parse(value).astimezone(pytz.timezone('Europe/Moscow'))

    @staticmethod
    def process_duration(value):
        """
        Из стартрека приходят в формате
        https://en.wikipedia.org/wiki/ISO_8601#Durations

        Так как они сами преобразуют рабочие дни, то мы это
        значение просто преобразуем в человекочитаемый вид
        """

        if not value:
            return

        match = isoduration.ISO8601_PERIOD_REGEX.match(value)
        if match is None:
            return

        units_order = (
            'years',
            'months',
            'weeks',
            'days',
            'hours',
            'minutes',
            'seconds',
        )

        parts = []
        for unit in units_order:
            value = match.groupdict()[unit]
            if value:
                parts.append(value.lower())
        return ' '.join(parts)
