import logging

from ..issue_processor import IssueProcessor

logger = logging.getLogger(__name__)

STARTREK_CLOSED_STATUS_KEY = 'closed'


class TicketType:
    STARTREK = 1

    CONST_2_STR = {
        STARTREK: 'startrek',
    }

    @staticmethod
    def str(v):
        return TicketType.CONST_2_STR[v]


class TicketDetails(object):
    """Хранит расширенные данные о тикете (Jira или Startrek).
    ticket_type должен быть равен значению из wf.wom.ticket.TicketType."""

    def __init__(self, id, summary, assignee_login, status, description, ticket_type, is_closed=False):
        self.id = id
        self.summary = summary
        self.assignee_login = assignee_login
        self.status = status
        self.description = description
        self.ticket_type = ticket_type
        self.is_closed = is_closed

    def __repr__(self):
        opts = (
            self.id,
            self.summary.encode('utf-8') if self.summary else 'None',
            self.assignee_login.encode('utf-8') if self.assignee_login else 'None',
            self.status.encode('utf-8') if self.status else 'None',
            self.description.encode('utf-8') if self.description else 'None',
            TicketType.str(self.ticket_type),
        )
        return 'Tiket:\nid=%s\nsummary=%s\nassignee_login=%s\nstatus=%s\ndescription=%s\nticket_type=%s\n' % opts


class WikiIssueLinkProcessor(IssueProcessor):
    expected_st_fields = (
        'id',
        'status',
        'summary',
        'assignee',
        'description',
    )

    @classmethod
    def process(cls, issue_data):
        try:
            status_key = issue_data['status'].key
            return TicketDetails(
                id=issue_data['id'],
                summary=issue_data['summary'],
                assignee_login=cls.process_user(issue_data, 'assignee'),
                status=status_key.capitalize(),
                description=issue_data['description'],
                ticket_type=TicketType.STARTREK,
                is_closed=status_key == STARTREK_CLOSED_STATUS_KEY,
            )
        except (KeyError, ValueError) as exc:
            logger.exception(exc)
