"""
    Base class for management commands that process users from django.contrib.auth
"""

from django.contrib.auth import get_user_model
from django.core.management.base import BaseCommand

User = get_user_model()


class UserCommand(BaseCommand):
    def add_arguments(self, parser):
        super(UserCommand, self).add_arguments(parser)
        parser.add_argument(
            '--dismissed',
            action='store_true',
            dest='only_dismissed',
            default=False,
            help='Run command only for dismissed users',
        )
        parser.add_argument(
            '--not-dismissed',
            action='store_true',
            dest='only_not_dismissed',
            default=False,
            help='Run command only for not dismissed users',
        )
        parser.add_argument(
            '--staff',
            action='store_true',
            dest='only_with_staff',
            default=False,
            help='Run command only for users with Staff',
        )
        parser.add_argument(
            '--no-staff',
            action='store_true',
            dest='only_without_staff',
            default=False,
            help='Run command only for users without Staff',
        )
        parser.add_argument(
            '--bypass', action='store_true', dest='bypass', default=False, help='Don\'t call handle_user()'
        )
        parser.add_argument('usernames', nargs='*', help='Usernames')

    def handle(self, *args, **options):
        constrains = []
        if options['usernames']:
            users = User.objects.filter(username__in=options['usernames'])
        else:
            users = User.objects.all()
        if options['only_dismissed']:
            users = users.filter(staff__is_dismissed=True)
            constrains.append('dismissed')
        if options['only_not_dismissed']:
            users = users.filter(staff__is_dismissed=False)
            constrains.append('not dismissed')
        if options['only_with_staff']:
            users = users.filter(staff__isnull=False)
            constrains.append('with Staff')
        if options['only_without_staff']:
            users = users.filter(staff__isnull=True)
            constrains.append('without Staff')
        num_users = users.count()
        if num_users == 0:
            print('No users found for your constraints. So I\'m doing nothing.')
            return
        print(
            'Going to handle {num_users} users {constrains}'.format(
                num_users=num_users, constrains=' '.join('*%s*' % cstr for cstr in constrains)
            )
        )
        confirmation = input('[Y/n]>')
        if confirmation not in 'Yy':
            print('You said {0!r}. So I\'m doing nothing.'.format(confirmation))
            return
        for user in users:
            if int(options['verbosity']) > 1:
                self.print_username_once(user)
            if not options['bypass']:
                self.handle_user(user, **options)
        print('Done. Bye.')

    def print_username_once(self, user):
        if not hasattr(self, '_users'):
            self._users = set()
        if user not in self._users:
            self._users.add(user)
            print(user.username)

    def handle_user(self, user, **options):
        raise NotImplementedError()
