from dataclasses import dataclass
from typing import Optional

from wiki.utils.dict_keypath import value_by_path
from wiki.utils.url import is_valid_url, is_wiki_url


class NodeType:
    HEAD = 'womHeading'
    HEAD_MD = 'heading'
    ACTION = 'womAction'
    LINK = 'link'
    WOMLINK = 'womLink'


class ActionName:
    ANCHOR = 'anchor'
    ANCHOR_SHORT = 'a'


def action_link(node) -> Optional[str]:
    for param in ['page', 'href']:
        if link := value_by_path(f'params.{param}', node):
            return link


class AST:
    def __init__(self, tree):
        self.tree = tree

    def filter(self, node_types):
        def traverse(node):
            if node.get('type', None) in node_types:
                yield node
            children = node.get('children', None)
            if children is not None:
                for child_node in children:
                    for deep_child_node in traverse(child_node):
                        yield deep_child_node

        return traverse(self.tree)

    def wiki_links(self):

        links = {node.get('url') for node in self.filter({NodeType.WOMLINK, NodeType.LINK})}

        # Добавляем урлы из инклюдов
        included_links = set()
        for action_node in self.filter({NodeType.ACTION}):
            node_name = action_node.get('name', '')
            if node_name in {'include', 'grid'}:
                if link := action_link(action_node):
                    included_links.add(link)

        links |= included_links

        external_links = {link for link in links if is_valid_url(link) and not is_wiki_url(link)}
        wiki_links = links - external_links
        return wiki_links

    def get_section(self, section_id: int):
        section = Section(section_id)
        for head_node in self.filter({NodeType.HEAD, NodeType.HEAD_MD}):
            section_local = head_node.get('section_local', -1)
            depth = head_node.get('depth', None)
            if section_local == section_id:
                section.pos_start = value_by_path('position.start.offset', head_node)
                section.pos_heading_end = value_by_path('position.end.offset', head_node)
                section.depth = depth
            elif section.depth is not None and depth <= section.depth:
                section.pos_end = value_by_path('position.start.offset', head_node)
                return section

        section.pos_end = value_by_path('position.end.offset', self.tree)
        return section

    def get_anchor(self, name: str):
        anchor = Anchor(name=name)
        for action_node in self.filter({NodeType.ACTION}):
            if action_node.get('name') not in {ActionName.ANCHOR, ActionName.ANCHOR_SHORT}:
                continue

            anchor_name = action_node.get('params', {}).get('name')
            if anchor_name == name:
                anchor.pos_start = value_by_path('position.start.offset', action_node)
                anchor.pos_end = value_by_path('position.end.offset', action_node)
                anchor.depth = action_node.get('depth', None)
                return anchor

        return anchor


@dataclass
class Section:
    section_id: int  # Порядковый номер секции
    pos_start: int = None  # Позиция начала секции
    pos_end: int = None  # Позиция конца секции
    pos_heading_end: int = None  # Позиция конца заголовка секции
    depth: int = None  # Глубина заголовка


@dataclass(slots=True, kw_only=True)
class Anchor:
    name: int
    pos_start: int = None
    pos_end: int = None
    depth: int = None
