import hashlib
import logging
import waffle

from django.conf import settings
from django.core.cache import caches
from json import dumps, loads
from wiki.unistat.metrics import WFAAS_RPS_METRIC

from wiki.api_core.waffle_switches import USE_LOCAL_WFAAS
from wiki.utils import api_request
from wiki.utils.errors import ApiRequestBadRequest, ApiRequestFailed, ApiRequestForbidden, ApiRequestWorkerFatal
from wiki.utils.tvm2 import get_service_ticket, parse_user_ticket
from wiki.utils.wfaas.ast import AST
from wiki.utils.wfaas.errors import WfaasAccessDenied, WfaasApiRequestError

logger = logging.getLogger(__name__)

WFAAS_CACHE = caches['wfaas_cache']


def get_wfaas_client(**kwargs):
    if settings.WFAAS_USE_TEST_CLIENT:
        return WfaasTestClient()

    return WfaasClient(**kwargs)


class WfaasClient:
    def __init__(self, timeout=25, max_retries=2):
        self.timeout = timeout
        self.max_retries = max_retries

    @staticmethod
    def _make_cache_key(endpoint, request_data, user_ticket=None):
        uid = '_'
        if user_ticket:
            parsed_data = parse_user_ticket(user_ticket)
            if parsed_data:
                uid = parsed_data.default_uid

        content_hex = hashlib.md5(dumps(request_data).encode()).hexdigest()
        return f'{endpoint}:{content_hex}:{uid}'

    def _call_wfaas(self, endpoint, request_data, user_ticket=None):
        cache_key = self._make_cache_key(endpoint, request_data, user_ticket)
        cached_data = WFAAS_CACHE.get(cache_key)
        if cached_data:
            logger.debug('[WFAAS] return cached data')
            return cached_data

        try:
            WFAAS_RPS_METRIC.increase()
            wfaas_host = settings.WFAAS_LOCAL_HOST if waffle.switch_is_active(USE_LOCAL_WFAAS) else settings.WFAAS_HOST
            response = api_request.post(
                url=f'http://{wfaas_host}/api/.{endpoint}',
                json=request_data,
                service_ticket=get_service_ticket(settings.WFAAS_TVM2_CLIENT_ID),
                user_ticket=user_ticket,
                headers={'Content-Type': 'application/json'},
                timeout=self.timeout,
                max_retries=self.max_retries,
                remote_name='WFAAS',
            )
            response_data = response.json()
        except ApiRequestBadRequest as e:
            logger.error(f'wfaas bad request {e.data}')
            raise WfaasApiRequestError()
        except ApiRequestForbidden as e:
            logger.error(f'wfaas access denied {e.data}')
            raise WfaasAccessDenied()
        except (ApiRequestWorkerFatal, ApiRequestFailed):
            raise WfaasApiRequestError()

        WFAAS_CACHE.set(cache_key, response_data, settings.WFAAS_CACHE_TIMEOUT)
        return response_data

    @staticmethod
    def _request_data(body, wfaas_settings=None):
        request_data = {'raw': body}
        if wfaas_settings is not None:
            request_data['settings'] = wfaas_settings
        return request_data

    def raw_to_ast(self, body, wfaas_settings=None,  user_ticket=None):
        request_data = self._request_data(body, wfaas_settings)
        wfaas_response = self._call_wfaas('raw_to_ast', request_data=request_data, user_ticket=user_ticket)
        return AST(wfaas_response['result']['ast'])

    def raw_to_html(self, body, wfaas_settings=None, user_ticket=None):
        request_data = self._request_data(body, wfaas_settings)
        wfaas_response = self._call_wfaas('raw_to_html', request_data, user_ticket=user_ticket)
        return wfaas_response['result']['html']

    def raw_to_bemjson(self, body, wfaas_settings=None, user_ticket=None):
        request_data = self._request_data(body, wfaas_settings)
        wfaas_response = self._call_wfaas('raw_to_bemjson', request_data, user_ticket=user_ticket)
        return wfaas_response['result']['bemjson']

    def yfm_to_html(self, body, wfaas_settings=None, user_ticket=None):
        request_data = self._request_data(body, wfaas_settings)
        wfaas_response = self._call_wfaas('yfm_to_html', request_data, user_ticket=user_ticket)
        return wfaas_response['result']['html']


class WfaasTestClient(WfaasClient):
    def _call_wfaas(self, endpoint, request_data, user_ticket):
        from intranet.wiki.tests.wiki_tests.common.data_helper import read_test_asset

        fixture_by_method = {
            'raw_to_ast': 'wfaas_ast_mock.json',
            'raw_to_bemjson': 'wfaas_bemjson_mock.json',
        }

        fixture = loads(read_test_asset(fixture_by_method[endpoint]))
        return fixture[request_data['raw']]
