import logging
import socket
from urllib.parse import urlparse, urlunparse

from django.http import Http404, HttpResponse

logger = logging.getLogger('wiki.xaccel_utils')

# путь, по которому настроен внутренний redirec у nginx
ACCEL_REDIRECT_SLUG = '/.storage/?fileurl='


def check_address_in_url(url):
    """
    Функция проверяет, что в url содержится ip-адрес хоста. Если нет - пытается разрешить из hostname
    Нужна, т.к. nginx требует при указании прокси указывать именно ip.

    Поскольку мы в Qloud, который ipv6-only, то вовзращать можно только ipv6-адрес.
    """
    purl = urlparse(url)

    hostname = purl.hostname
    # простая проверка на то, что hostname - валидный ip-адрес
    try:
        socket.inet_pton(socket.AF_INET6, hostname)
    except socket.error:
        # пытаемся разрезолвить самостоятельно
        logger.warning('no IP address in url %s, trying to resolve', url)
        addr_info = socket.getaddrinfo(purl.hostname, socket.IPPROTO_TCP)
        # addr_info содержит что-то подобное:
        # [(10, 1, 6, '', ('2a02:6b8:0:3400::1:159', 6, 0, 0)),
        # (10, 2, 17, '', ('2a02:6b8:0:3400::1:159', 6, 0, 0)),
        # (10, 3, 0, '', ('2a02:6b8:0:3400::1:159', 6, 0, 0))]
        hostname = '[%s]' % addr_info[0][4][0]

    if purl.port:
        hostname = '{host}:{port}'.format(host=hostname, port=purl.port)

    return urlunparse((purl.scheme, hostname, purl.path, '', '', ''))


def accel_redirect_to_download(django_model, content_type):
    """
    Вернуть данные из хранилища, используя X-Accel-Redirect заголовок для nginx.

    В конфиге nginx должно висеть примерно такое правило:

    location /.storage/ {
        internal;
        proxy_pass $arg_fileurl;
        proxy_hide_header Content-Type;
    }

    url, с которого выкачивается файл, берется из query_param 'fileurl' в хедере 'X-Accel-Redirect'
    "proxy_hide_header Content-Type;" гарантирует, что пользователю вернется Content-Type, выставленный django

    @type django_model: models.Model
    @type content_type: unicode
    @raises: Http404
    @return: HttpResponse
    """
    if not django_model.mds_storage_id:
        logger.error('object "%s" id=%d has no storage id', django_model.__class__, django_model.id)
        raise Http404()

    # url файла, с hostname замененным на ip адрес хранилища, для корректного
    # внутреннего редиректа nginx
    storage_url = django_model.mds_storage_id.storage.url(django_model.mds_storage_id.name)
    if not storage_url:
        logger.error(
            'Cannot convert storage id "%s" of object "%s" pk=%s to URL',
            django_model.mds_storage_id,
            django_model.__class__,
            django_model.id,
        )
        raise Http404()

    url = check_address_in_url(storage_url)

    # болванка ответа, nginx пройдет по урлу в хэдере X-Accel-Redirect
    # и отдаст пользователю уже файл. При этом важно не забыть
    # выставить директиву "proxy_hide_header Content-Type;", иначе content-type
    # будет контролировать уже хранилище
    response = HttpResponse(content='', content_type=content_type)
    response['X-Accel-Redirect'] = ACCEL_REDIRECT_SLUG + url
    response['X-Accel-Buffering'] = 'no'  # не кэшируем отданные файлы

    return response
