import calendar
import logging
from collections import defaultdict

from django.conf import settings
from wiki.intranet.models import Staff

from wiki.org import get_org
from wiki.pages.access import get_raw_access, interpret_raw_access
from wiki.pages.dao.actuality import get_actual_external_link_urls, get_linked_actual_pages_supertags
from wiki.pages.models.page import Page
from wiki.pages.models.consts import ACTUALITY_STATUS
from wiki.personalisation.user_cluster import is_in_user_cluster

logger = logging.getLogger(__name__)

IS_INTRANET = getattr(settings, 'IS_INTRANET', False)


def acl_list(page, access):
    result = []
    if access['is_owner']:
        if page.get_authors():
            result.extend([author.username for author in page.get_authors()])
    else:
        for group in access['groups']:
            for member in group_members_for_template(group):
                result.append(member)
        for user in access['users']:
            # wiki.intranet.models.Staff
            if hasattr(user, 'login'):
                result.append(user.login)
            # django.contrib.auth.models.User
            elif hasattr(user, 'username'):
                result.append(user.username)
            else:
                raise TypeError('Unknown user type')
    return result


def group_members_for_template(group):
    """Вернуть генератор представлений членов группы для шаблона

    Group|str -> unicode|str
    """
    for member in group.get_all_members():
        yield member.username


def group_for_template(group):
    """Вернуть группу для записи в шаблон.

    Group|str -> str
    """
    if settings.IS_BUSINESS:
        from wiki.users.models import GROUP_TYPES

        return {'id': group.dir_id, 'type': GROUP_TYPES[group.group_type]}
    else:
        return str(group.id)


# Возвращает список групп
def acl_groups(page, access):
    result = []
    if (access['is_common'] or access['is_common_wiki']) and settings.IS_INTRANET:
        result.append(str(settings.YANDEX_GROUP_ID))
    for group in access['groups']:
        result.append(group_for_template(group))
    return result


# Возвращает список юзеров (если они были перечислены отдельно от групп)
def acl_users(page, access):
    result = []
    if page.get_authors():
        result.extend([author.username for author in page.get_authors()])
    for user in access['users']:
        # wiki.intranet.models.Staff
        if hasattr(user, 'login'):
            result.append(user.login)
        # django.contrib.auth.models.User
        elif hasattr(user, 'username'):
            result.append(user.username)
        # wiki.pages.models.User
        else:
            raise TypeError('Unknown user type')
    return result


def xstr(s):
    return '' if s is None else s


class PageEditor(object):
    """
    Редактировавший страницу пользователь.
    """

    login = None
    edited_times = None
    # таймстемп
    last_edition_date = 0

    def __init__(self, login=None, edited_times=0, last_edition_date=0):
        self.edited_times = edited_times
        self.login = login
        self.last_edition_date = last_edition_date

    def __str__(self):
        return '{login}: {times} times, last at {last}'.format(
            login=self.login, times=self.edited_times, last=self.last_edition_date
        )

    def __eq__(self, other):
        return all(
            getattr(self, attr_name) == getattr(other, attr_name)
            for attr_name in (
                'login',
                'edited_times',
                'last_edition_date',
            )
        )

    __repr__ = __str__


def get_editors_from_db(page):
    """
    Вернуть список пар (логин, дата правки) про страницу.
    """
    return page.revision_set.values_list('author__username', 'created_at')


def get_editors_of_page_for_yaserver(page):
    """
    Вернуть список редактировавших страницу в неопределенном порядке.

    @rtype: list
    """
    result = defaultdict(PageEditor)
    page_editions = get_editors_from_db(page)
    for edition in page_editions:
        result_edition = result[edition[0]]
        result_edition.edited_times += 1
        result_edition.last_edition_date = (
            max(result_edition.last_edition_date, edition[1]) if result_edition.last_edition_date else edition[1]
        )
    # теперь переведем дату в таймстемп и проставим поле login объекту PageEditor
    for login, page_editor in result.items():
        page_editor.login = login
        page_editor.last_edition_date = calendar.timegm(page_editor.last_edition_date.timetuple())
    return list(result.values())


def yaserver_context(page):
    """
    Добавить в контекст ответа яндекс.серверу мета-данные,
    выводимые в <head> страницы
    """
    ctx = {}
    access = interpret_raw_access(get_raw_access(page.tag))

    # доступы
    ctx['is_public'] = int(access['is_common'] or access['is_common_wiki'])
    ctx['acl_groups_whitelist'] = acl_groups(page, access)
    ctx['acl_users_whitelist'] = acl_users(page, access)

    # время создания/изменения в unix time
    ctx['creation_time'] = calendar.timegm(page.created_at.timetuple())

    ctx['modification_time'] = calendar.timegm(page.modified_at_for_index.timetuple())

    # количестов потомков страницы
    ctx['descendants_count'] = page.descendants.count()

    # является ли страница пользовательским кластером
    ctx['is_user_cluster'] = int(is_in_user_cluster(page))

    editors = get_editors_of_page_for_yaserver(page)
    # количество авторов, изменявших страницу
    ctx['modifiers_count'] = len(editors)
    ctx['editors'] = editors

    # количество ссылок на страницу с других страниц
    ctx['linked_from_count'] = page.links_to.all().count()
    # количество раз, когда страницу помещали в избранное
    # TODO: WIKI-9661
    ctx['favorited_count'] = 0

    # оффициальность страницы
    ctx['is_official'] = int(page.is_official)

    ctx['is_documentation'] = int(page.is_documentation)

    # владелец страницы
    if page.owner:
        ctx['owner_login'] = page.owner.username
        try:
            staff = page.owner.staff
            ctx['owner_full_name'] = '{0} {1}'.format(xstr(staff.first_name), xstr(staff.last_name)).strip()
            ctx['owner_full_name_en'] = '{0} {1}'.format(xstr(staff.first_name_en), xstr(staff.last_name_en)).strip()
        except Staff.DoesNotExist as exc:
            logger.error('staff object does not exist for user %s: %s', ctx['owner_login'], exc)

    authors = []
    if page.get_authors():
        for author in page.get_authors():
            author_data = {'author_login': author.username}
            try:
                staff = author.staff
                author_data['author_full_name'] = '{0} {1}'.format(
                    xstr(staff.first_name), xstr(staff.last_name)
                ).strip()
                author_data['author_full_name_en'] = '{0} {1}'.format(
                    xstr(staff.first_name_en), xstr(staff.last_name_en)
                ).strip()
            except Staff.DoesNotExist as exc:
                logger.error('staff object does not exist for user %s: %s', author.username, exc)

            authors.append(author_data)

    ctx['authors'] = authors

    # собрать родителей, даже "виртуальных"
    supertags = page.get_supertags_chain()
    query_set = Page.objects.filter(supertag__in=supertags, org=get_org()).values_list('supertag', 'title')
    title_map = {supertag: title for supertag, title in query_set}
    all_parents = []
    for supertag in supertags:
        all_parents.append((supertag, title_map[supertag] if supertag in title_map else None))
    ctx['parents'] = all_parents

    if page.actuality_status == ACTUALITY_STATUS.obsolete:
        ctx['not_actual_since'] = page.actuality_marked_at.strftime('%s')
        ctx['actual_pages'] = ','.join(get_linked_actual_pages_supertags(page.id))
        ctx['actual_urls'] = ','.join(get_actual_external_link_urls(page.id))
    # не для всех вики у нас есть NODEJS_FRONTEND_HOST
    ctx['frontend_host'] = getattr(settings, 'NGINX_HOST', 'undefined for this wiki')

    return ctx
