from __future__ import absolute_import

import logging

from redis import Redis
from redis.exceptions import LockError
from redis.lock import Lock
from redis.sentinel import Sentinel, MasterNotFoundError
from typing import Optional
from ylock.base import BaseManager, BaseLock
from django.conf import settings
from django.core.cache import caches

logger = logging.getLogger(__name__)

SOCKET_TIMEOUT = 0.1


class Manager(BaseManager):
    default_block_timeout = 0

    def __init__(
        self,
        password=None,
        hosts=None,
        sentinel_port=26379,
        redis_cluster_name=None,
        default_lock_lifetime=None,
        prefix='',
    ):
        self.hosts = hosts
        self.sentinel_port = sentinel_port
        self.password = password
        self.redis_cluster_name = redis_cluster_name
        self.default_lock_lifetime = default_lock_lifetime
        self.prefix = prefix
        self.sentinel = None  # type: Optional[Sentinel]
        self.master = None  # type: Optional[Redis]

    def get_master(self, refresh=False):
        if self.sentinel is None or self.master is None or refresh:
            self.master = None
            for host in self.hosts:
                self.sentinel = Sentinel([(host, self.sentinel_port)], socket_timeout=SOCKET_TIMEOUT)
                try:
                    master = self.sentinel.master_for(self.redis_cluster_name, password=self.password)
                    master.ping()
                    self.master = master
                    break
                except MasterNotFoundError:
                    continue
            if self.master is None:
                raise Exception("Can't find master. Make sure that password, hosts and port are correct")

        return self.master

    def lock(self, name, timeout=None, block=True, block_timeout=5, lock_class=None, sleep=0.1, thread_local=True):
        """
        Return a new Lock object using key ``name`` that mimics
        the behavior of threading.Lock.

        If specified, ``timeout`` indicates a maximum life for the lock.
        By default, it will remain locked until release() is called.

        ``sleep`` indicates the amount of time to sleep per loop iteration
        when the lock is in blocking mode and another client is currently
        holding the lock.

        ``block_timeout`` indicates the maximum amount of time in seconds to
        spend trying to acquire the lock. A value of ``None`` indicates
        continue trying forever. ``blocking_timeout`` can be specified as a
        float or integer, both representing the number of seconds to wait.

        ``lock_class`` forces the specified lock implementation.

        ``thread_local`` indicates whether the lock token is placed in
        thread-local storage. By default, the token is placed in thread local
        storage so that a thread only sees its token, not a token set by
        another thread. Consider the following timeline:

            time: 0, thread-1 acquires `my-lock`, with a timeout of 5 seconds.
                     thread-1 sets the token to "abc"
            time: 1, thread-2 blocks trying to acquire `my-lock` using the
                     Lock instance.
            time: 5, thread-1 has not yet completed. redis expires the lock
                     key.
            time: 5, thread-2 acquired `my-lock` now that it's available.
                     thread-2 sets the token to "xyz"
            time: 6, thread-1 finishes its work and calls release(). if the
                     token is *not* stored in thread local storage, then
                     thread-1 would see the token value as "xyz" and would be
                     able to successfully release the thread-2's lock.

        In some use cases it's necessary to disable thread local storage. For
        example, if you have code where one thread acquires a lock and passes
        that lock instance to a worker thread to release later. If thread
        local storage isn't disabled in this case, the worker thread won't see
        the token set by the thread that acquired the lock. Our assumption
        is that these cases aren't common and as such default to using
        thread local storage."""

        if block is None:
            raise ValueError('Block arg must be bool')

        if settings.USE_MDB_REDIS:
            lock = caches['ylock_redis'].lock(
                name,
                timeout=timeout or self.default_lock_lifetime,
                sleep=sleep,
                block_timeout=block_timeout,
                thread_local=thread_local,
            )
            return RedisLock(self, lock, block)

        lock = Lock(
            self.get_master(),
            name=self.prefix + '/' + name,
            timeout=timeout or self.default_lock_lifetime,
            sleep=sleep,
            blocking=block,
            blocking_timeout=block_timeout,
            thread_local=thread_local,
        )
        return RedisLock(self, lock, lock.blocking)

    def lock_from_context(self, context):
        """Возвращает лок, восстановленный из контекста"""
        lock = self.get_master().lock(**context)
        return RedisLock(self, lock, lock.blocking)


class RedisLock(BaseLock):
    """
    Враппер над нативным редисовым Lock
    @todo - обработать кейс если мастер уже не мастер
    """

    def __init__(self, mgr, redis_lock, blocking):
        self._mgr = mgr  # type: Manager
        self._lock = redis_lock  # type: Lock
        self._was_acquired = False
        self._blocking = blocking

    def acquire(self):
        """Берет лок. Возврщает True если удалось"""
        q = self._lock.acquire(self._blocking)
        self._was_acquired = self._was_acquired or q
        return q

    def release(self):
        """Отпускает лок. Возвращает True если удалось"""
        try:
            if self._was_acquired:
                # был захвачен хотя бы раз. не так точно как self._lock.locked() но -1 запрос в редьку
                self._lock.release()
            self._was_acquired = False
            return True
        except LockError:
            logger.exception('Can not release %s lock' % self._lock.name)
            return False

    def __enter__(self):
        return self.acquire()

    def __exit__(self, exc_type, exc_value, traceback):
        self.release()

    def check_acquired(self):
        return self._lock.locked()

    def get_context(self):
        """Возвращает контекст из которго можно потом востановить лок.
        Должен пиклиться.
        """
        return {
            'name': self._lock.name,
            'timeout': self._lock.timeout,
            'sleep': self._lock.sleep,
            'blocking': self._lock.blocking,
            'blocking_timeout': self._lock.blocking_timeout,
            'thread_local': self._lock.thread_local,
        }
