# flake8: noqa

import re
import threading
import time
import urllib.error
import urllib.parse
import urllib.request

import django
from django.core.handlers.wsgi import WSGIHandler
from django.urls import Resolver404, resolve

from wiki.env import set_env_type

set_env_type()

django.setup()

_application = WSGIHandler()

API_NAMESPACES = ('frontend', 'api_v1', 'svc')


def is_api_url(path):
    """
    Вернуть True если урл является одним из урлов их публичного RESTful АПИ вики.

    @rtype: bool
    """
    try:
        return any(namespace in resolve(path).namespaces for namespace in API_NAMESPACES)
    except Resolver404:
        return False


if threading.active_count() > 1:
    print()
    print('**** Several threads are active before uwsgi fork() ****')
    print('This will likely to cause deadlock if GIL is in one of the threads before fork()')
    print('What threads do we have:')

    for thread in threading.enumerate():
        print(('Thread: %s' % thread))

    print("I will sleep, hoping GIL will get back to main thread and we won't go deadlocked. No promises though")
    print()
    time.sleep(1)


# Тут мы фиксим проблему, связанную с потерей символов прямой слэш в url, являющихся частью имени,
# а не разделительным символом. Потеря происходит в процессе двойного энкодинга в разных сервисах: Nginx и Django и
# не воспроизводится локально, например, в тестах.
# см. статью http://codeinthehole.com/writing/django-nginx-wsgi-and-encoded-slashes/
def application(environ, start_response):
    request_uri_no_query_params = environ['REQUEST_URI'].split('?')[0]
    if is_api_url(request_uri_no_query_params):
        # Строки в environ должны быть в кодировке ISO-8859-1
        # https://www.python.org/dev/peps/pep-3333/#unicode-issues
        path = urllib.parse.unquote(request_uri_no_query_params).encode('utf-8').decode('iso-8859-1')

        # убираем лишние слеши в пути
        environ['PATH_INFO'] = re.sub('/{2,}', '/', path)

    return _application(environ, start_response)
