
from django.contrib.auth import get_user_model
from django.test import TestCase
from ujson import loads

from wiki.sync.connect import OPERATING_MODE_NAMES
from wiki.sync.connect.models import OperatingMode, Organization
from wiki.billing import RESTRICTION_NAMES
from wiki.billing.logic import update_org_stats
from wiki.files.models import File
from wiki.notifications.models import PageEvent
from wiki.pages.models import Page
from wiki.utils.timezone import now
from intranet.wiki.tests.wiki_tests.common.unittest_base import BaseApiTestCase

STATUS_ENABLED = Organization.ORG_STATUSES.enabled


class UpdateOrgStatsTest(TestCase):
    def test_stats_update(self):
        OperatingMode.objects.get_or_create(name=OPERATING_MODE_NAMES.free)
        User = get_user_model()
        user = User()
        user.save()

        org_1 = Organization(name='1', dir_id='1', label='1', status=STATUS_ENABLED)
        org_1.save()

        page_1_org_1 = Page(supertag='smth', tag='smth', org=org_1, last_author=user, modified_at=now())
        page_1_org_1.save()
        page_1_org_1.authors.add(user)

        page_2_org_1 = Page(
            supertag='sandbox/kek',
            tag='sandbox/kek',
            org=org_1,
            last_author=user,
            modified_at=now(),
            is_autogenerated=True,
        )
        page_2_org_1.save()
        page_2_org_1.authors.add(user)

        page_3_org_1 = Page(
            supertag='redirect',
            tag='redirect',
            org=org_1,
            last_author=user,
            modified_at=now(),
        )
        page_3_org_1.redirects_to = page_1_org_1  # редиректы не учитываются в статистике
        page_3_org_1.save()
        page_3_org_1.authors.add(user)

        file_1_page_1_org_1 = File(
            page=page_1_org_1,
            user=user,
            size=7 * (1 << 20),
            url='1',
        )
        file_1_page_1_org_1.save()

        file_2_page_1_org_1 = File(
            page=page_1_org_1,
            user=user,
            size=3 * (1 << 20),
            url='2',
        )
        file_2_page_1_org_1.save()

        file_1_page_2_org_1 = File(
            page=page_2_org_1,
            user=user,
            size=4 * (1 << 20),
            url='1',
        )
        file_1_page_2_org_1.save()

        file_2_page_2_org_1 = File(
            page=page_2_org_1,
            user=user,
            size=6 * (1 << 20),
            url='2',
        )
        file_2_page_2_org_1.save()

        org_2 = Organization(name='2', dir_id='2', label='2', status=STATUS_ENABLED)
        org_2.save()

        page_1_org_2 = Page(supertag='smth', tag='smth', org=org_2, last_author=user, modified_at=now())
        page_1_org_2.save()
        page_1_org_2.authors.add(user)

        page_2_org_2 = Page(supertag='smth_else', tag='smth_else', org=org_2, last_author=user, modified_at=now())
        page_2_org_2.save()
        page_2_org_2.authors.add(user)

        org_3 = Organization(name='3', dir_id='3', label='3', status=STATUS_ENABLED)
        org_3.save()

        # У организации 3 статистика не отличается от дефолтной,
        # для нее обновления не будет.
        update_org_stats()

        org_1 = Organization.objects.get(id=org_1.id)
        org_2 = Organization.objects.get(id=org_2.id)
        org_3 = Organization.objects.get(id=org_3.id)

        self.assertEqual(
            {
                RESTRICTION_NAMES.org_page_num: 1,
                RESTRICTION_NAMES.org_attach_size_mb: 20,
            },
            org_1.orgstatistics.statistics,
        )
        self.assertEqual(
            {
                RESTRICTION_NAMES.org_page_num: 2,
                RESTRICTION_NAMES.org_attach_size_mb: 0,
            },
            org_2.orgstatistics.statistics,
        )
        self.assertEqual(
            {
                RESTRICTION_NAMES.org_page_num: 0,
                RESTRICTION_NAMES.org_attach_size_mb: 0,
            },
            org_3.orgstatistics.statistics,
        )

        file_3_page_1_org_1 = File(
            page=page_1_org_1,
            user=user,
            size=6 * (1 << 20),
            url='3',
        )
        file_3_page_1_org_1.save()

        page_3_org_2 = Page(
            supertag='sandbox/nano_gleb',
            tag='sandbox/nano_gleb',
            org=org_2,
            last_author=user,
            modified_at=now(),
            is_autogenerated=True,
        )
        page_3_org_2.save()
        page_3_org_2.authors.add(user)

        file_1_page_1_org_2 = File(
            page=page_1_org_2,
            user=user,
            size=42,  # мелкий файл, не отразится на мегабайтах
            url='1',
        )
        file_1_page_1_org_2.save()

        update_org_stats()

        org_1 = Organization.objects.get(id=org_1.id)

        self.assertEqual(
            {
                RESTRICTION_NAMES.org_page_num: 1,
                RESTRICTION_NAMES.org_attach_size_mb: 26,
            },
            org_1.orgstatistics.statistics,
        )

        page_1_org_1.status = 0
        page_1_org_1.save()
        file_2_page_2_org_1.status = 0
        file_2_page_2_org_1.save()

        update_org_stats()

        # У организации 1 была удалена одна страница с ее
        # файлами и отдельно один файл, статистика должна учеть это.

        org_1 = Organization.objects.get(id=org_1.id)

        self.assertEqual(
            {
                RESTRICTION_NAMES.org_page_num: 0,
                RESTRICTION_NAMES.org_attach_size_mb: 4,
            },
            org_1.orgstatistics.statistics,
        )

    def test_is_limits_exceeded_update(self):
        OperatingMode.objects.get_or_create(name=OPERATING_MODE_NAMES.free)
        User = get_user_model()
        user = User()
        user.save()

        org_1_mode = OperatingMode(
            limits={
                RESTRICTION_NAMES.org_page_num: 2,
                RESTRICTION_NAMES.org_attach_size_mb: 2,
            },
            name='org_1_mode',
        )
        org_1_mode.save()

        org_2_mode = OperatingMode(
            limits={
                RESTRICTION_NAMES.org_page_num: 2,
                RESTRICTION_NAMES.org_attach_size_mb: 2,
            },
            name='org_2_mode',
        )
        org_2_mode.save()

        org_1 = Organization(name='1', dir_id='1', label='1', status=STATUS_ENABLED)
        org_1.save()

        org_1.mode = org_1_mode
        org_1.save()

        page_1_org_1 = Page(supertag='smth', tag='smth', org=org_1, last_author=user, modified_at=now())
        page_1_org_1.save()
        page_1_org_1.authors.add(user)

        page_2_org_1 = Page(
            supertag='topor',
            tag='topor',
            org=org_1,
            last_author=user,
            modified_at=now(),
        )
        page_2_org_1.save()
        page_2_org_1.authors.add(user)

        page_3_org_1 = Page(
            supertag='tapok',
            tag='tapok',
            org=org_1,
            last_author=user,
            modified_at=now(),
        )
        page_3_org_1.save()
        page_3_org_1.authors.add(user)

        file_1_page_1_org_1 = File(
            page=page_1_org_1,
            user=user,
            size=3 * (1 << 20),
        )
        file_1_page_1_org_1.save()

        org_2 = Organization(name='2', dir_id='2', label='2', status=STATUS_ENABLED)
        org_2.save()

        org_2.mode = org_2_mode
        org_2.save()

        page_1_org_2 = Page(supertag='smth', tag='smth', org=org_2, last_author=user, modified_at=now())
        page_1_org_2.save()
        page_1_org_2.authors.add(user)

        page_2_org_2 = Page(
            supertag='topor',
            tag='topor',
            org=org_2,
            last_author=user,
            modified_at=now(),
        )
        page_2_org_2.save()
        page_2_org_2.authors.add(user)

        file_1_page_1_org_2 = File(
            page=page_1_org_2,
            user=user,
            size=2 * (1 << 20),
        )
        file_1_page_1_org_2.save()

        update_org_stats()

        org_1 = Organization.objects.get(id=org_1.id)
        org_2 = Organization.objects.get(id=org_2.id)

        self.assertTrue(org_1.orgstatistics.is_limits_exceeded)
        self.assertFalse(org_2.orgstatistics.is_limits_exceeded)

        org_1_mode.limits = {
            RESTRICTION_NAMES.org_page_num: 3,
            RESTRICTION_NAMES.org_attach_size_mb: 3,
        }
        org_1_mode.save()

        org_2_mode.limits = {
            RESTRICTION_NAMES.org_page_num: 1,
            RESTRICTION_NAMES.org_attach_size_mb: 2,
        }
        org_2_mode.save()

        update_org_stats()

        # У организаций не изменилась статистика,
        # но изменились тарифные планы так, что теперь
        # у организации 1 не превышен лимит,
        # а у организации 2 превышен.

        org_1 = Organization.objects.get(id=org_1.id)
        org_2 = Organization.objects.get(id=org_2.id)

        self.assertFalse(org_1.orgstatistics.is_limits_exceeded)
        self.assertTrue(org_2.orgstatistics.is_limits_exceeded)


class LimitsCheckTest(BaseApiTestCase):
    def setUp(self):
        super(LimitsCheckTest, self).setUp()
        self.setUsers()
        self.client.login('thasonic')
        self.user = self.user_thasonic

    def test_page_num_limit(self):
        org = Organization(name='1', dir_id='1', label='1', status=Organization.ORG_STATUSES.enabled)
        org.save()

        org.mode.limits = {
            RESTRICTION_NAMES.org_page_num: 1,
            RESTRICTION_NAMES.org_attach_size_mb: 3,
        }
        org.mode.save()

        self.user.orgs.add(org)

        page_data = {'title': 'title', 'body': 'body'}

        request_url = '{api_url}/{page_tag}'.format(api_url=self.api_url, page_tag='page1')
        response = self.client.post(request_url, data=page_data)

        self.assertEqual(200, response.status_code)

        update_org_stats()

        request_url = '{api_url}/{page_tag}'.format(api_url=self.api_url, page_tag='page2')
        response = self.client.post(request_url, data=page_data)

        # Уже превысили лимит в одну страницу
        self.assertEqual(402, response.status_code)
        self.assertEqual('billing.limits:Page num limit 1 pages', response.json()['error']['message'][0])

        request_url = '{api_url}/{page_tag}'.format(api_url=self.api_url, page_tag='page1')
        response = self.client.post(request_url, data=page_data)

        # Отредактировать существующую страницу можно
        self.assertEqual(200, response.status_code)

        org.mode.limits = {
            RESTRICTION_NAMES.org_page_num: 2,
            RESTRICTION_NAMES.org_attach_size_mb: 3,
        }
        org.mode.save()

        request_url = '{api_url}/{page_tag}'.format(api_url=self.api_url, page_tag='page2')
        response = self.client.post(request_url, data=page_data)

        # Лимит повысился на 1, теперь можно создать страницу
        self.assertEqual(200, response.status_code)

        update_org_stats()

        request_url = '{api_url}/{page_tag}'.format(api_url=self.api_url, page_tag='page3')
        response = self.client.post(request_url, data=page_data)

        self.assertEqual(402, response.status_code)
        self.assertEqual('billing.limits:Page num limit 2 pages', response.json()['error']['message'][0])

        self.client.delete('{api_url}/{page_supertag}'.format(api_url=self.api_url, page_supertag='page1'))

        update_org_stats()

        request_url = '{api_url}/{page_tag}'.format(api_url=self.api_url, page_tag='page3')
        response = self.client.post(request_url, data=page_data)

        # Одну страницу удалили, теперь можно создать еще одну
        self.assertEqual(200, response.status_code)

    def test_attach_size_limit(self):
        org = self.org_42

        org.mode.limits = {
            RESTRICTION_NAMES.org_page_num: 0,
            RESTRICTION_NAMES.org_attach_size_mb: 1,
        }
        org.mode.save()

        self.user.orgs.add(org)

        def upload_file(file_name, data):
            from intranet.wiki.tests.wiki_tests.common.utils import encode_multipart_formdata

            content_type, body = encode_multipart_formdata(
                [('someField', 'somedata')], [('filefield', file_name, data)]
            )

            response = self.client.post('/_api/frontend/.upload', data=body, content_type=content_type)

            return response

        response = upload_file('1.txt', 'data')
        self.assertEqual(200, response.status_code)

        org.orgstatistics.statistics[RESTRICTION_NAMES.org_attach_size_mb] = 2
        org.orgstatistics.save()

        response = upload_file('2.txt', 'data')
        self.assertEqual(402, response.status_code)
        self.assertEqual('billing.limits:Files size limit 0.001 Gb', response.json()['error']['message'][0])

        org.mode.limits = {
            RESTRICTION_NAMES.org_page_num: 0,
            RESTRICTION_NAMES.org_attach_size_mb: -1,
        }
        org.mode.save()

        response = upload_file('2.txt', 'data')
        self.assertEqual(200, response.status_code)

    def test_max_rev_num_limit(self):
        org = Organization(name='1', dir_id='1', label='1', status=Organization.ORG_STATUSES.enabled)
        org.save()

        org.mode.limits = {
            RESTRICTION_NAMES.org_page_num: 10,
            RESTRICTION_NAMES.user_max_rev_num: 4,
        }
        org.mode.save()

        self.user.orgs.add(org)

        # События обратно хронологически:
        # edit edit watch watch edit mark_actual create

        edit_page_url = '{api_url}/{page_tag}'.format(api_url=self.api_url, page_tag='tag')

        self.client.post(edit_page_url, data={'title': 'title', 'body': 'rev 0'})  # создаст create event

        page = Page.active.get(supertag='tag')

        PageEvent(
            page=page,
            timeout=now(),
            author=self.user,
            event_type=PageEvent.EVENT_TYPES.mark_actual,
            meta={},
        ).save()

        self.client.post(edit_page_url, data={'title': 'title', 'body': 'rev 1'})  # создаст edit event

        PageEvent(
            page=page,
            timeout=now(),
            author=self.user,
            event_type=PageEvent.EVENT_TYPES.watch,
            meta={},
        ).save()

        PageEvent(
            page=page,
            timeout=now(),
            author=self.user,
            event_type=PageEvent.EVENT_TYPES.watch,
            meta={},
        ).save()

        self.client.post(edit_page_url, data={'title': 'title', 'body': 'rev 2'})  # создаст edit event
        self.client.post(edit_page_url, data={'title': 'title', 'body': 'rev 3'})  # создаст edit event

        # Грузим все события с лимитом 4, загрузятся последние 4 - edit, edit, watch, watch
        request_url = '{api_url}/{page_supertag}/.events'.format(api_url=self.api_url, page_supertag='tag')
        response = self.client.get(request_url)

        self.assertEqual(200, response.status_code)
        data = loads(response.content)['data']

        self.assertEqual(data['total'], 7)
        self.assertEqual(data['limit_exceeded'], True)
        self.assertEqual(data['limit'], 4)
        self.assertEqual(len(data['data']), 4)
        self.assertEqual(data['data'][0]['event_type'], 'edit')
        self.assertEqual(data['data'][1]['event_type'], 'edit')
        self.assertEqual(data['data'][2]['event_type'], 'watch')
        self.assertEqual(data['data'][3]['event_type'], 'watch')

        # Грузим события edit с лимитом 4, загрузятся только два, так как
        # третье событие edit не влезает в последние 4 события.
        request_url = '{api_url}/{page_supertag}/.events?event_types={types}'.format(
            api_url=self.api_url, page_supertag='tag', types='edit'
        )
        response = self.client.get(request_url)

        self.assertEqual(200, response.status_code)
        data = loads(response.content)['data']

        self.assertEqual(data['total'], 3)
        self.assertEqual(data['limit_exceeded'], True)
        self.assertEqual(data['limit'], 4)
        self.assertEqual(len(data['data']), 2)
        self.assertEqual(data['data'][0]['event_type'], 'edit')
        self.assertEqual(data['data'][1]['event_type'], 'edit')

        # Грузим события mark_actual с лимитом 4, вернется пустой список, так как
        # событие mark_actual не влезает в последние 4 события.
        request_url = '{api_url}/{page_supertag}/.events?event_types={types}'.format(
            api_url=self.api_url, page_supertag='tag', types='mark_actual'
        )
        response = self.client.get(request_url)

        self.assertEqual(200, response.status_code)
        data = loads(response.content)['data']

        self.assertEqual(data['total'], 1)
        self.assertEqual(data['limit_exceeded'], True)
        self.assertEqual(data['limit'], 4)
        self.assertEqual(len(data['data']), 0)

        # Грузим все события с лимитом 4 и смещением 3, вернется только одно событие.
        request_url = '{api_url}/{page_supertag}/.events?start=3'.format(
            api_url=self.api_url, page_supertag='tag'  # , types='mark_actual' ??
        )
        response = self.client.get(request_url)

        self.assertEqual(200, response.status_code)
        data = loads(response.content)['data']

        self.assertEqual(data['total'], 7)
        self.assertEqual(data['limit_exceeded'], True)
        self.assertEqual(data['limit'], 4)
        self.assertEqual(len(data['data']), 1)
        self.assertEqual(data['data'][0]['event_type'], 'watch')

        # Грузим события edit с лимитом 4 и смещением 2, вернется пустой список,
        # так как третье событие edit не влезает в последние 4 события.
        request_url = '{api_url}/{page_supertag}/.events?start=2&event_types={types}'.format(
            api_url=self.api_url, page_supertag='tag', types='edit'
        )
        response = self.client.get(request_url)

        self.assertEqual(200, response.status_code)
        data = loads(response.content)['data']

        self.assertEqual(data['total'], 3)
        self.assertEqual(data['limit_exceeded'], True)
        self.assertEqual(data['limit'], 4)
        self.assertEqual(len(data['data']), 0)

        # Грузим события edit с лимитом 4 и смещением 3, вернется пустой список
        # с признаком limit_exceeded=False, так как событий edit всего 3.
        request_url = '{api_url}/{page_supertag}/.events?start=3&event_types={types}'.format(
            api_url=self.api_url, page_supertag='tag', types='edit'
        )
        response = self.client.get(request_url)

        self.assertEqual(200, response.status_code)
        data = loads(response.content)['data']

        self.assertEqual(data['total'], 3)
        self.assertEqual(data['limit_exceeded'], False)
        self.assertEqual(data['limit'], 4)
        self.assertEqual(len(data['data']), 0)

        # Грузим все события с лимитом 4 и лимитом отображения 4, вернутся четыре события
        # с признаком limit_exceeded=False.
        request_url = '{api_url}/{page_supertag}/.events?limit=4'.format(
            api_url=self.api_url, page_supertag='tag'  # , types='mark_actual' ???
        )
        response = self.client.get(request_url)

        self.assertEqual(200, response.status_code)
        data = loads(response.content)['data']

        self.assertEqual(data['total'], 7)
        self.assertEqual(data['limit_exceeded'], False)
        self.assertEqual(data['limit'], 4)
        self.assertEqual(len(data['data']), 4)
        self.assertEqual(data['data'][0]['event_type'], 'edit')
        self.assertEqual(data['data'][1]['event_type'], 'edit')
        self.assertEqual(data['data'][2]['event_type'], 'watch')
        self.assertEqual(data['data'][3]['event_type'], 'watch')

        org.mode.limits = {
            RESTRICTION_NAMES.org_page_num: 10,
            RESTRICTION_NAMES.user_max_rev_num: 7,
        }
        org.mode.save()

        # Грузим все события с лимитом 7, все события помещаются.
        request_url = '{api_url}/{page_supertag}/.events'.format(api_url=self.api_url, page_supertag='tag')
        response = self.client.get(request_url)

        self.assertEqual(200, response.status_code)
        data = loads(response.content)['data']

        self.assertEqual(data['total'], 7)
        self.assertEqual(data['limit_exceeded'], False)
        self.assertEqual(data['limit'], 7)
        self.assertEqual(len(data['data']), 7)
        self.assertEqual(data['data'][0]['event_type'], 'edit')
        self.assertEqual(data['data'][1]['event_type'], 'edit')
        self.assertEqual(data['data'][2]['event_type'], 'watch')
        self.assertEqual(data['data'][3]['event_type'], 'watch')
        self.assertEqual(data['data'][4]['event_type'], 'edit')
        self.assertEqual(data['data'][5]['event_type'], 'mark_actual')
        self.assertEqual(data['data'][6]['event_type'], 'create')
