import re
from io import StringIO
from lxml.etree import Element
import sys

sys.setrecursionlimit(2000)

from lxml import etree

from intranet.wiki.tools.wikiclient import translit

MAX_TITLE_LENGTH = 255

YELLOW_PRE = """<#<table class="wikisnippet_alert" style="border-left:4px solid #ffde40; margin:20px; width:90%; background-color:#fffbe6">
<tr valign="middle" style="vertical-align: middle;"><td style="vertical-align: middle; padding:20px; 20px 0 0px; text-align:left;">#>
"""

RED_PRE = """<#<table class="wikisnippet_alert" style="border-left:4px solid #e45d50; margin:20px; width:90%; background-color:#fbe8e6">
<tr valign="middle" style="vertical-align: middle;"><td style="vertical-align: middle; padding:20px; 20px 0 0px; text-align:left;">#>
"""

GREEN_PRE = """<#<table class="wikisnippet_alert" style="border-left:4px solid #97B25C; margin:20px; width:90%; background-color:#d9e3c3;">
<tr valign="middle" style="vertical-align: middle;"><td style="vertical-align: middle; padding:20px; 20px 0 0px; text-align:left;">#>
"""

GRAY_PRE = """<#<table class="wikisnippet_alert" style="border-left:4px solid #aaa; margin:20px; width:90%; background-color:#f1f1f1;">
<tr valign="middle" style="vertical-align: middle;"><td style="vertical-align: middle; padding:20px; 20px 0 0px; text-align:left;">#>
"""

NOTE_PRE = """<#<table class="wikisnippet_alert" style="border-left:4px solid #ffde40; margin:20px; width:90%; background-color:#fffbe6">
<tr valign="middle" style="vertical-align: middle;"><td style="vertical-align: middle; padding:20px; 20px 0 0px; text-align:left;">#>
"""
NOTE_POST = '<#</td></tr></table>#>\n'

ESCAPE = {
    '&gt;': '$E#!gt!#E$',
    '&lt;': '$E#!lt!#E$',
    '&amp;': '$E#!amp!#E$',
}

# &lt;a href=&quot;url&quot;&gt;МРТ&lt;/a&gt;
XML_WRAPPER = """\
<?xml version="1.1"?>
<html>
<body>
%s
</body>
</html>"""

list_tags = {
    'ol': '1. ',
    'ul': '* ',
    'ac:task-list': '* ',
}


def is_local_src(src):
    return src.startswith('attachments/')


INLINE_MARKS = {
    'strong': 'b',
    'b': 'b',
    'em': 'i',
    'i': 'i',
    'u': 'u',
}


def mk(a=None, b=None, strip=False):
    def f(ctx, node):

        if node.tag in INLINE_MARKS:
            # Для правки такого:
            # <strong>last_paid,&nbsp;<strong>next_pay,&nbsp;<strong>nex_sum
            # ,&nbsp;<strong>pay_role</strong>&nbsp;</strong></strong></strong>
            if INLINE_MARKS[node.tag] in ctx.inline_flags:
                return ctx.convert(node)
            ctx.inline_flags.add(INLINE_MARKS[node.tag])

        c = ctx.convert(node)

        if node.tag in INLINE_MARKS:
            ctx.inline_flags.remove(INLINE_MARKS[node.tag])

        w0 = ''
        w1 = ''
        if strip:
            wb = r'^(\s*)'
            we = r'(\s*)$'
            if q := re.findall(wb, c):
                w0 = q[0]

            if q := re.findall(we, c):
                w1 = q[0]
            c = c.strip()

        if c is None or c == '':
            return f'{w0}{w1}'

        return f'{w0}{a or ""}{c}{b or ""}{w1}'

    return f


mk_bold = mk('**', '**', True)

"""
"strong": mk('**', '**', True),
"b": mk('**', '**', True),
"em": mk('_', '_', True),
"u": mk('__', '__', True),
"""


def fixtail(fn):
    def fn2(ctx, node):
        i = fn(ctx, node)
        if i.startswith('***__') and i.endswith('__***'):
            i = '**__*' + i[5:-5] + '*__**'
        if i.startswith('***') and i.endswith('***'):
            i = '**_' + i[3:-3] + '_**'

        return i

    return fn2


def iframe_macro(ctx, node):
    """
        <ac:structured-macro ac:name="iframe" ac:schema-version="1" data-layout="default" ac:local-
    id="80aae67c-9a7c-4fe3-92d6-0d778c902969" ac:macro-id="be8b980cace8a5924d3b4bef11a6020e"><ac:parameter
    ac:name="src"><ri:url ri:value="https://calendar.google.com/calendar/embed?src=c_tghfchebhv36keuguna00op9mc%40group.cale
    ndar.google.com&amp;ctz=Europe%2FMoscow" /></ac:parameter><ac:parameter ac:name="width">700</ac:parameter><ac:parameter
    ac:name="height">500</ac:parameter></ac:structured-macro>
        {{iframe src="http://адресСтраницы" width=700px height=600px frameborder=0 scrolling=no}}
    """
    params = _params_as_dict(ctx, node)

    args = ['frameborder=0']

    if 'width' in params:
        args.append(f'width={params["width"]}px')
    if 'height' in params:
        args.append(f'height={params["height"]}px')

    try:
        src = (
            [q for q in node.getchildren() if q.tag == 'ac:parameter' and q.get('ac:name') == 'src'][0]
            .getchildren()[0]
            .get('ri:value')
        )
        args.append(f'src="{src}"')
    except:
        ctx.addwarn('??Bad iframe??')
        return '??Bad iframe??'
    args = ' '.join(args)
    return f'{{{{iframe {args}}}}}\n'


def include_macro(ctx, node):
    """
        <ac:structured-macro ac:name="include" ac:schema-version="1" data-layout="default" ac:local-id="bb5d0cf2-2ed2-4199-8403-2ef69b4da84a" ac:macro-
    id="239717a01245971c90272b6f38add760"><ac:parameter ac:name=""><ac:link><ri:page ri:content-title="Перечень уведомлений,
    формируемых по событиям" ri:version-at-save="7" /></ac:link></ac:parameter></ac:structured-macro>
    """
    try:
        tgt = node.getchildren()[0].getchildren()[0].getchildren()[0]
    except:
        ctx.addwarn('Bad include')
        return '??Bad include macro??'

    if tgt.tag == 'ri:page':
        space = tgt.get('ri:space-key') or ctx.current_space
        title = tgt.get('ri:content-title')
        page_slug = ctx.page_title_lut.get(space, {}).get(title, None)

        if page_slug:
            return f'((/{page_slug} {title}))\n'
        else:
            ctx.addwarn(f'Broken include to {space}: {title}')
            return f'??[Broken include to {space}: {title}]??'

    ctx.addwarn('Bad include')
    return '??Bad include macro??'


def viewfile_macro(ctx, node):
    for c in node.getchildren():
        if c.tag == 'ac:parameter' and c.get('ac:name') == 'name':
            return link(ctx, c)

    ctx.addwarn('Bad viewfile')
    return '??Bad viewfile macro??'


def noteblock_macro(ctx, node):
    """
      info = gray
      tip = green
      note = yellow
      warning = red
      Macro name: info/tip/note/warning
      Macro body: Accepts rich text.

      <ac:structured-macro ac:name="info">
    <ac:parameter ac:name="icon">false</ac:parameter>
    <ac:parameter ac:name="title">This is my title</ac:parameter>
    <ac:rich-text-body>
      <p>
            <span>This is </span> <em>important</em> <span> information.</span>
          </p>
        </ac:rich-text-body>
      </ac:structured-macro>
    """
    coll = {
        'note': [YELLOW_PRE, NOTE_POST],
        'warning': [RED_PRE, NOTE_POST],
        'info': [GRAY_PRE, NOTE_POST],
        'tip': [GREEN_PRE, NOTE_POST],
    }
    params = _params_as_dict(ctx, node)
    name = node.get('ac:name')
    if params.get('title'):
        return (
            coll[name][0]
            + '**'
            + params.get('title', '').strip()
            + '**\n\n'
            + params.get('ac:rich-text-body')
            + '\n'
            + coll[name][1]
        )

    return coll[name][0] + params.get('ac:rich-text-body') + '\n' + coll[name][1]


def _pad_li(ctx, data, put_mark=True):
    li_sign = ctx.li_stack[-1]
    return '\n'.join(
        [((li_sign if line_id == 0 and put_mark else ' ' * len(li_sign))) + line for line_id, line in enumerate(data)]
    )


def li(ctx, node: Element):
    return ctx.convert(node)
    # data = ctx.convert(node).split('\n')
    # return _pad_li(ctx, data)


def _params_as_dict(ctx, node: Element):
    q = {}
    for n in node.getchildren():
        if n.tag == 'ac:parameter':
            q[n.get('ac:name', 'unknown')] = ctx.convert(n)
        if n.tag == 'ac:rich-text-body':
            q['ac:rich-text-body'] = ctx.convert(n)
    return q


def macro_status(ctx, node: Element):
    """
    <ac:structured-macro ac:name="status" ac:schema-version="1" ac:macro-id="641d2660-4ece-46ff-9c17-a3667aee7ec9">
    <ac:parameter ac:name="title">в работе</ac:parameter><ac:parameter ac:name="colour">Green</ac:parameter>
    </ac:structured-macro>

    """
    params = _params_as_dict(ctx, node)

    return f'({params.get("title")})'


def li_task(ctx, node: Element):
    status = 'unknown'
    body = ''

    for child in node.getchildren():
        if child.tag == 'ac:task-status':
            status = ctx.convert(child)
        if child.tag == 'ac:task-body':
            body = ctx.convert(child)

    if status == 'complete':
        status = '[x]'
    else:
        status = '[]'

    data = f'{status} {body}'

    return _pad_li(ctx, [data])


def bq(ctx, node: Element):
    ctx.bq += 1
    data = ctx.convert(node).strip().split('\n')
    v = '\n'.join(['>' * ctx.bq + line for line in data]) + '\n'
    ctx.bq -= 1

    return v


def ul_ol(ctx, node: Element):
    ctx.li_stack.append(list_tags[node.tag])
    li = []
    for child in node.getchildren():
        data = ctx.parse_node(child).strip('\n').split('\n')
        li.append(_pad_li(ctx, data, child.tag == 'li'))
    ctx.li_stack.pop()
    x = '\n'.join(li)
    out = '\n' + x + '\n'
    if len(ctx.li_stack) == 0:
        out += '\n'
    return out


def nop(ctx, node):
    return ctx.convert(node)


def time(ctx, node):
    # <time datetime="2020-10-19" />
    return node.get('datetime', '?')


def table(ctx, node):
    return '\n#|\n' + ctx.convert(node) + '\n|#\n'


def tr(ctx, node):
    return '\n||\n' + ctx.convert(node, merge_with=' | ') + '\n||\n'


def td_th(ctx, node):
    return ctx.convert(node).replace('|', r'\|')


def img(ctx, node):
    src = node.get('src')
    if src:
        return f'![]({src})' + '\n'
    else:
        return ''


def image(ctx, node):
    # <ac:image ac:align="center" ac:layout="center" ac:original-height="503" ac:original-width="858">
    # <ri:attachment ri:filename="image-20210215-061255.png" ri:version-at-save="1" />

    target = node.getchildren()

    if len(target) < 1:
        return '??Broken IMG??'

    if target[0].tag == 'ri:attachment':
        filename = target[0].get('ri:filename')
        path = f'file:/{ctx.current_page_slug}/{translit(filename)}'
    elif target[0].tag != 'ri:url':
        path = target[0].get('ri:value')
    else:
        return '??Broken IMG??'

    width = node.get('ac:width', '0')
    height = node.get('ac:height', '0')

    if width or height:
        path = f'{width}x{height}:{path}'

    return path + '\n'


def link(ctx, node):
    tgt = None
    body = None

    for c in node.getchildren():
        if c.tag.startswith('ri:'):
            tgt = c
        if c.tag == 'ac:link-body':
            body = ctx.convert(c)

    if tgt is None:
        ctx.addwarn('Broken link')
        return '??Broken link??'

    if tgt.tag == 'ri:user':
        key = tgt.get('ri:userkey')
        if key in ctx.user_resolve:
            return f'@{ctx.user_resolve[key]}'

        return f'??[Unknown User key={tgt.get("ri:userkey")}]??'

    if tgt.tag == 'ri:attachment':
        filename = tgt.get('ri:filename')
        return f'file:/{ctx.current_page_slug}/{translit(filename)}'

    if tgt.tag == 'ri:page':
        space = tgt.get('ri:space-key') or ctx.current_space
        title = tgt.get('ri:content-title')
        page_slug = ctx.page_title_lut.get(space, {}).get(title, None)
        body = body or title

        if page_slug:
            return f'((/{page_slug} {body}))'
        else:
            ctx.addwarn(f'Broken link {space}: {title}')
            return f'??[Broken link {space}: {title}]??'

    # <ri:page ri:space-key="prodaction" ri:content-title="Введение" ri:version-at-save="1" />
    # <ri:user ri:userkey="ff8080815c5b3224015c5ddec8d70001" />

    ctx.addwarn(f'Link to {tgt.tag}')
    return f'??Link to {tgt.tag}??'


def ignore(ctx, node):
    return ''


def anchor(ctx, node):
    # todo -- convert!
    # https://docdoc.atlassian.net/wiki/spaces/DP/pages/2581070123
    href = node.get('href')
    return f'(({href} {ctx.convert(node)}))'


def jira(ctx, node):
    # ac:parameter ac:name="key">DD-30406</ac:parameter>
    for child in node.getchildren():
        if child.tag == 'ac:parameter' and child.get('ac:name') == 'key':
            return child.text
    return '??Broken JIRA macro??'


def pagetree(ctx, node):
    return '{{tree}}\n\n'


def macroanchor(ctx, node):
    d = _params_as_dict(ctx, node)
    return '{{ anchor name="%s" }}' % d.get('', 'smth')


def expand(ctx, node):
    # ac:parameter ac:name="key">DD-30406</ac:parameter>
    title = 'Readmore'
    body = ''
    for child in node.getchildren():
        if child.tag == 'ac:parameter' and child.get('ac:name') == 'title':
            title = ctx.convert(child)
        if child.tag == 'ac:rich-text-body':
            body = ctx.convert(child)

    return '<{%s\n%s}>\n' % (title, body)


def notimplemented(ctx, node):
    return f'??Not implemented {node.tag}??\n'


def maybecolor(ctx, node):
    if 'color' in ctx.inline_flags:
        return ctx.convert(node)

    wf_colors = [
        ('red', (0xCC, 0, 0)),
        ('green', (0, 0x88, 0)),
        ('blue', (0, 0x44, 0xBB)),
        ('grey', (0x80, 0x80, 0x80)),
        ('yellow', (0xFF, 0xD7, 0)),
        ('violet', (0xFF, 0, 0xFF)),
        ('orange', (0xFF, 0x88, 0)),
        ('cyan', (0, 0xFF, 0xFF)),
        ('black', (0, 0, 0)),
    ]
    s = re.findall(r'rgb\(\s*(\d+)\s*,\s*(\d+)\s*,\s*(\d+)\s*\)', node.get('style', ''))
    pre = ''
    post = ''
    if len(s) == 1:
        vec = [int(i) for i in s[0]]
        best_match = None
        best_fit = None
        for name, color in wf_colors:
            dist = sum([(color[i] - vec[i]) * (color[i] - vec[i]) for i in range(3)])
            if best_match is None or dist < best_fit:
                best_match = name
                best_fit = dist
        if best_match != 'black':
            pre = f'!!({best_match})' if best_match != 'red' else '!!'
            post = '!!'

    ctx.inline_flags.add('color')
    data = ctx.convert(node)
    ctx.inline_flags.remove('color')

    # eject whitespaces
    wb = r'^(\s*)'
    we = r'(\s*)$'
    w0 = ''
    w1 = ''
    if q := re.findall(wb, data):
        w0 = q[0]
    if q := re.findall(we, data):
        w1 = q[0]

    data = data.strip()

    if data is None or data == '':
        return w0 + w1

    return w0 + pre + data + post + w1


def macrocode(ctx, node):
    lang = ''
    data = ''
    for child in node.getchildren():
        if child.tag == 'ac:parameter' and child.get('ac:name') == 'language':
            lang = child.text
        if child.tag == 'ac:plain-text-body':
            data = nop(ctx, child)
    return f'```{lang}\n{data}\n```\n\n'


class ConfluenceXmlParser:
    def __init__(self, page_id_lut, page_title_lut, user_resolve, paranoid=False):
        self.page_title_lut = page_title_lut  # space -> title -> pk
        self.page_id_lut = page_id_lut  # space -> title -> pk
        self.user_resolve = user_resolve  # space -> title -> pk
        self.current_page_slug = 'temp'
        self.paranoid = paranoid
        self.toks = []
        self.bq = 0
        self.li_stack = []
        self.inline_flags = set()
        self.warnings = set()
        self.table_depth = 0

        self.macro_handlers = {
            'toc': mk('{{toc}}\n'),
            'code': macrocode,
            'details': nop,
            'jira': jira,
            'expand': expand,
            'anchor': macroanchor,
            'children': pagetree,
            'pagetree': pagetree,
            'status': macro_status,
            'lucidchart': notimplemented,
            'info': noteblock_macro,
            'tip': noteblock_macro,
            'note': noteblock_macro,
            'warning': noteblock_macro,
            'view-file': viewfile_macro,
            'include': include_macro,
            'iframe': iframe_macro,
        }

        self.tag_handlers = {
            'a': anchor,
            'ac:parameter': ignore,
            'blockquote': bq,
            'table': table,
            'colgroup': ignore,
            'tbody': nop,
            'ac:task-id': nop,
            'ac:task-status': nop,
            'ac:task-body': nop,
            'ac:placeholder': mk('_++!!(gray)', '!!++_ ', True),
            'thead': nop,
            'div': nop,
            'ac:layout': nop,
            'ac:layout-cell': nop,
            'ac:layout-section': nop,
            'ac:adf-node': nop,
            'ac:adf-extension': nop,
            'ac:adf-attribute': nop,
            'ac:adf-fallback': nop,
            'ac:adf-content': nop,
            'ri:url': nop,
            'span': maybecolor,
            'hr': mk('---\n', ''),
            'tr': tr,
            'img': img,
            'td': td_th,
            'th': td_th,
            'p': mk('', '\n\n'),
            'strong': fixtail(mk('**', '**', True)),
            'b': fixtail(mk('**', '**', True)),
            'em': fixtail(mk('*', '*', True)),
            'u': fixtail(mk('__', '__', True)),
            'del': mk('~~', '~~', True),
            'sup': mk('^^', '^^', True),
            'sub': mk('++', '++', True),
            'code': mk('##', '##', True),
            's': nop,
            'small': mk('++', '++', True),
            'br': mk('<br/>'),
            'li': li,
            'ac:task': li_task,
            'ol': ul_ol,
            'ul': ul_ol,
            'ac:task-list': ul_ol,
            'html': nop,
            'body': nop,
            'time': time,
            'plain-text-body': mk('```\n', '\n```\n\n'),
            'pre': mk('```\n', '\n```\n\n'),
            'ac:inline-comment-marker': nop,
            'ac:structured-macro': self.parse_macro,
            'ac:rich-text-body': nop,
            'ac:link': link,
            'ac:image': image,
            'ac:note': nop,
            'ac:emoticon': ignore,
        }
        for i in range(1, 7):
            self.tag_handlers[f'h{i}'] = mk('#' * i + ' ', '\n')

    def parse(self, data, current_page_slug, current_space):
        self.current_page_slug = current_page_slug
        self.current_space = current_space
        data = (XML_WRAPPER % data).replace(']] >', ']]>').replace('] ]>', ']]>').replace('&nbsp;', ' ')

        for original, escaped in ESCAPE.items():
            data = data.replace(original, escaped)

        parser = etree.XMLParser(recover=True, resolve_entities=True)

        root = etree.parse(StringIO(data), parser=parser).getroot()

        ret = self.parse_node(root).strip()
        for original, escaped in ESCAPE.items():
            ret = ret.replace(escaped, original)

        return ret

    def parse_macro(self, ctx, elm):
        macro_id = elm.get('ac:name')
        if macro_id not in self.macro_handlers:
            self.addwarn(f'Missing macro {macro_id}')
            if self.paranoid:
                return f'??Unhandled macro: {macro_id}?? ' + self.convert(elm)
            else:
                return self.convert(elm)
        else:
            return self.macro_handlers[macro_id](self, elm)

    def parse_node(self, elm):
        if elm.tag not in self.tag_handlers:
            self.addwarn(f'Missing handler for <{elm.tag}>')
            if self.paranoid:
                return f'`<{elm.tag}>` ' + self.convert(elm) + f' `</{elm.tag}>`'
            else:
                return self.convert(elm)
        else:
            return self.tag_handlers[elm.tag](self, elm)

    def addwarn(self, msg):
        self.warnings.add(msg)

    def convert(self, elm, merge_with=None):
        merge_with = merge_with or ''
        return (elm.text or '') + merge_with.join([self.parse_node(q) + (q.tail or '') for q in elm.getchildren()])
