import logging
from pprint import pprint
from tqdm import tqdm
from django.db import connection

from wiki.sync.connect.dir_client import DirClient, ServiceStatusAction, OrganizationAlreadyDeletedError
from django.conf import settings

from wiki.sync.connect.models import Organization

logger = logging.getLogger(__name__)

dir_client = DirClient()

stats = []
dir_ids = []

def process(from_m = 2, to_m = 3):
    with connection.cursor() as cursor:
        RAW_SQL = f"""
        WITH orgs as (
            SELECT id, dir_id
            FROM dir_data_sync_organization as org
            WHERE cloud_id IS NOT NULL AND created > NOW() - interval '{to_m} MONTH'
                                       AND created <= NOW() - interval '{from_m} MONTH'
        ),
        prepass as (
           SELECT orgs.dir_id as dir_id, p.supertag,
           CASE
               WHEN p.supertag ~* '^(users/([^/]*?)$|users/([^/]*?)/notes$|clusterusers$|homepage$|users$|sandbox|sandbox/(.*)$)' THEN COUNT (1)-1
               ELSE COUNT(1)
           END as non_robot_revisions,

           MAX(r.created_at) as created_at
           from pages_page as p

            JOIN pages_revision as r on p.id = r.page_id
            JOIN orgs ON p.org_id = orgs.id
            GROUP BY orgs.dir_id, p.supertag),

        filter_pass as  (
            SELECT dir_id, SUM(non_robot_revisions) as non_robot_revisions_total
            from prepass
            group by dir_id
         )

        SELECT dir_id, non_robot_revisions_total
            from filter_pass
            WHERE non_robot_revisions_total = 0;
        """

        cursor.execute(RAW_SQL)
        rows = cursor.fetchall()

    for dir_id, _ in tqdm(rows):
        if dir_id in dir_ids:
            print(f'Skipping {dir_id}...')
            continue
        try:
            dir_org = dir_client.get_organization(dir_id, fields=settings.DIRSYNC_FIELDS_ORGANIZATION_FOR_SYNC)
        except OrganizationAlreadyDeletedError:
            print(f'Skipping {dir_id}...')
            continue

        admin_uid = dir_org.get('admin_id')
        print(f'Disabling {dir_org["name"]} CloudId {dir_org["cloud_org_id"]} DirID {dir_id}')
        try:
            dir_client.change_service_status(dir_id, ServiceStatusAction.DISABLE, admin_uid)
        except Exception as e:
            print(f'Disabling {dir_org["name"]} CloudId {dir_org["cloud_org_id"]} DirID {dir_id} FAILED -- {e}')
            continue

        assert Organization.objects.filter(dir_id=dir_id).update(status='disabled') == 1
        stats.append((dir_org["name"], dir_org["cloud_org_id"], dir_id))
        dir_ids.append(dir_id)



