import logging
from json import loads
from typing import Tuple, Dict

import requests

from intranet.wiki.tools.wikiclient.utils import make_service_ticket
from intranet.wiki.tools.wikiclient.consts import ApiRequestFailed, AuthMode, B2BCreds

logger = logging.getLogger(__name__)


def slashify(path):
    if path is None or path == '':
        return path
    if path[-1] != '/':
        path = path + '/'
    if path[0] == '/':
        path = path[1:]

    return path


def safe_loads(s):
    try:
        return loads(s)
    except Exception:
        raise ValueError(f'Cant decode server response "{str(s)}"')


class E2eTvmRestApi:
    def use_api_v2_public(self):
        self.default_api = 'api/v2/public/'
        return self

    def use_api_v2_support(self):
        self.default_api = 'api/v2/support/'
        return self

    def as_user(self, creds: B2BCreds = None, org_id=None, uid=None, cloud_uid=None):
        if creds:
            self.org_id = creds.org_id
            self.user_uid = creds.uid
            self.cloud_uid = creds.cloud_uid

        else:
            self.org_id = org_id
            self.user_uid = uid
            self.cloud_uid = cloud_uid
        return self

    def tvm2_auth(self, tvm2_client=None, tvm2_server=None):
        if tvm2_client:
            self._tvm2_src = tvm2_client
        if tvm2_server:
            self._tvm2_dst = tvm2_server
        self.auth = AuthMode.TVM2
        return self

    def _tvm_auth_headers(self, headers):
        service_ticket = make_service_ticket(self._tvm2_src, self._tvm2_dst)
        headers.update(
            {
                'X-Ya-Service-Ticket': service_ticket,
            }
        )

    def oauth(self, oauth):
        self._oauth = oauth
        self.auth = AuthMode.OAUTH
        return self

    def _oauth_auth_headers(self, headers):
        headers.update(
            {
                'Authorization': f'OAuth {self._oauth}',
            }
        )

    auth_fn = {
        AuthMode.TVM2: _tvm_auth_headers,
        AuthMode.OAUTH: _oauth_auth_headers,
    }

    def __init__(self, host: str, default_api=''):
        self.host = slashify(host)
        self.default_api = slashify(default_api)
        self.auth = AuthMode.NONE
        self.org_id = None
        self.user_cloud_id = None
        self.user_uid = None

    def api_call(
        self,
        method='get',
        endpoint='me',
        json=None,
        data=None,
        params=None,
        plain=False,
        headers=None,
        files=None,
        timeout=10.0,
        parse_json=True,
        api=None,
    ) -> Tuple[int, Dict]:

        if endpoint[0] == '/':
            endpoint = endpoint[1:]

        func = safe_loads
        if not parse_json:
            func = lambda x: x

        headers = headers or {}
        default_content_type = 'application/json' if not plain else 'application/octet-stream'

        headers.update({'Content-Type': headers.get('Content-Type', default_content_type)})

        if self.org_id:
            headers['x-org-id'] = self.org_id
        if self.user_cloud_id:
            headers['x-cloud-uid'] = self.user_cloud_id
        if self.user_uid:
            headers['x-uid'] = self.user_uid

        if files:
            del headers['Content-Type']

        self.inject_headers(headers)

        attempt = 0
        MAX_ATTEMPTS = 3

        last_cause = 'Unset'

        while attempt < MAX_ATTEMPTS:
            attempt += 1
            if api is not None:
                url = self.host + slashify(api) + endpoint
            elif self.default_api is not None:
                url = self.host + self.default_api + endpoint
            else:
                url = self.host + endpoint

            try:
                resp = requests.request(
                    method,
                    url,
                    params=params,
                    json=json,
                    data=data,
                    headers=headers,
                    timeout=timeout,
                    files=files,
                    verify=False,
                )
                logger.info(f'[{resp.status_code}] {method} {url} {params} {data}')
            except requests.RequestException as e:
                last_cause = f'Request error {str(e)}'
                logger.exception('API Request error')
                continue

            if 200 <= resp.status_code <= 204:
                return resp.status_code, func(resp.content.decode())

            if 400 <= resp.status_code < 429:
                return resp.status_code, func(resp.content.decode())

            last_cause = f'Api Call: [{resp.status_code}] {resp.content}'
            logger.info(last_cause)

        logger.error(f'Out of attempts. Last error: {last_cause}')
        raise ApiRequestFailed(last_cause)

    def inject_headers(self, headers):
        if self.auth != AuthMode.NONE:
            self.auth_fn[self.auth](self, headers)
