"""
Транслитерирование русского текста.

@author: Eugene Voytitsky (voyt@yandex-team.ru)
"""

import re

macroses = {
    'вики': 'wiki',
    'вако': 'wacko',
    'вака': 'shwacko',
    'веб': 'web',
    'ланс': 'lance',
    'кукуц': 'kukutz',
    'мендокуси': 'mendokusee',
    'яремко': 'iaremko',
    'николай': 'nikolai',
    'алексей': 'aleksey',
    'анатолий': 'anatoly',
}
single_char_letters_from = [
    'а',
    'б',
    'в',
    'г',
    'д',
    'е',
    'з',
    'и',
    'к',
    'л',
    'м',
    'н',
    'о',
    'п',
    'р',
    'с',
    'т',
    'у',
    'ф',
    'х',
    'ъ',
    'ь',
    'ц',
    'ы',
    'А',
    'Б',
    'В',
    'Г',
    'Д',
    'Е',
    'З',
    'И',
    'К',
    'Л',
    'М',
    'Н',
    'О',
    'П',
    'Р',
    'С',
    'Т',
    'У',
    'Ф',
    'Х',
    'Ъ',
    'Ь',
    'Ц',
    'Ы',
]
single_char_letters_to = [
    'a',
    'b',
    'v',
    'g',
    'd',
    'e',
    'z',
    'i',
    'k',
    'l',
    'm',
    'n',
    'o',
    'p',
    'r',
    's',
    't',
    'u',
    'f',
    'x',
    '',
    '',
    'c',
    'y',
    'A',
    'B',
    'V',
    'G',
    'D',
    'E',
    'Z',
    'I',
    'K',
    'L',
    'M',
    'N',
    'O',
    'P',
    'R',
    'S',
    'T',
    'U',
    'F',
    'X',
    '',
    '',
    'C',
    'Y',
]
double_char_letters_from = ['й', 'ё', 'ж', 'ч', 'ш', 'щ', 'э', 'ю', 'я', 'Й', 'Ё', 'Ж', 'Ч', 'Ш', 'Щ', 'Э', 'Ю', 'Я']
double_char_letters_to = [
    'jj',
    'jo',
    'zh',
    'ch',
    'sh',
    'shh',
    'je',
    'ju',
    'ja',
    'Jj',
    'Jo',
    'Zh',
    'Ch',
    'Sh',
    'Shh',
    'Je',
    'Ju',
    'Ja',
]
single_chars_length = len(single_char_letters_from)
double_chars_length = len(double_char_letters_from)

trash_re = re.compile(r'[^A-Za-z0-9\.\/\-]')


def _translit(word, to_lower=True):
    # substitute macroses
    for macros_key in macroses:
        start = 0
        while True:
            position = word.lower().find(macros_key, start)
            start = position + 1
            if position > -1:
                if word[position].lower() == word[position]:
                    continue
                key_length = len(macros_key)

                if len(word) > position + key_length:
                    if word[position + key_length].lower() == word[position + key_length]:
                        continue

                macros_value = macroses[macros_key]
                word = word[:position] + macros_value.capitalize() + word[position + key_length :]
            else:
                break

    # substitute single char translit chars
    for i in range(single_chars_length):
        word = word.replace(single_char_letters_from[i], single_char_letters_to[i])

    # substitute double char translit chars
    for i in range(double_chars_length):
        word = word.replace(double_char_letters_from[i], double_char_letters_to[i])

    r = trash_re.sub('', word)
    return r.lower() if to_lower else r


def translit(tag, to_lower=True):
    """
    Транслитерировать все русские буквы и удалить все недопустимые символы в строке,
    включая начальный и конечный символ "/".
    Если to_lower=True, перевести результат в нижний регистр.
    """
    return _translit(tag, to_lower).strip('/')


def remove_trailing_dots(tag):
    """
    Удаляет из тега первые и последние точки на всех уровнях. Например, "aaa.1./.bbb.2." -> "aaa.1/bbb.2"
    Если в теге встречается "/./", это будет заменено на один слеш
    """
    return '/'.join(chunk.strip('.') for chunk in tag.split('/') if chunk.strip('.'))


def tag_to_supertag(tag):
    """
    Возвращает супертэг для тэга.

    @type tag: basestring
    @rtype: basestring
    """
    return translit(remove_trailing_dots(tag))


def normalize_slug(dirty_supertag: str):
    """
    Единственно верный способ приведения грязного supertag к чистому
    :param dirty_supertag:
    :return:
    """
    if dirty_supertag is None:
        return dirty_supertag
    return translit(remove_trailing_dots(dirty_supertag))
