import tqdm

from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_main_connection,
    get_meta_connection,
    get_shard_numbers,
    get_shard,
)
from intranet.yandex_directory.src.yandex_directory.core.task_queue.exceptions import DuplicatedTask
from collections import defaultdict, namedtuple
from intranet.yandex_directory.src.yandex_directory.core.maillist_check.tasks import MaillistsCheckTask
from intranet.yandex_directory.src.yandex_directory.core.models.group import UserGroupMembership
from intranet.yandex_directory.src.yandex_directory.core.models import (
    DomainModel,
    DepartmentModel,
    GroupModel,
    UserModel,
    UserMetaModel,
    RobotServiceModel,
    ResourceRelationModel,
)
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    get_domain_id_from_blackbox,
)
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    create_maillist,
    get_user_data_from_blackbox_by_login,
    get_user_data_from_blackbox_by_uid,
    get_organization_admin_uid,
)

Item = namedtuple('Item', ['id', 'shard', 'org_id', 'type', 'label', 'uid', 'domain', 'revision'])


def iterate_over_items(shard, chunk_size=100):
   query = """
   select departments.id, departments.org_id, uid, 'department', label, domains.name, revision_counters.revision
     from departments
     left join domains on departments.org_id = domains.org_id
                      and domains.owned = True
                      and domains.master = True
     left join revision_counters on departments.org_id = revision_counters.org_id
    where uid is not null
union all
   select      groups.id,      groups.org_id, uid, 'group',      label, domains.name, revision_counters.revision
     from groups
     left join domains on groups.org_id = domains.org_id
                      and domains.owned = True
                      and domains.master = True
     left join revision_counters on groups.org_id = revision_counters.org_id
    where uid is not null
   """
   count_query = """
   select count(*)
     from (
       select org_id, uid
         from departments
        where uid is not null
    union all
       select org_id, uid
         from groups
        where uid is not null
     ) as t
   """
   total = main_connection.execute(count_query).fetchall()[0][0]

   def make_iterator():
       with get_main_connection(shard=shard) as main_connection:
           cursor = main_connection.execute(query)

           rows = cursor.fetchmany(chunk_size)
           while rows:
               for row in rows:
                   yield row
               rows = cursor.fetchmany(chunk_size)

   return tqdm.tqdm(make_iterator(), total=total)



def fetch_data():
    data = defaultdict(list)
    for shard in get_shard_numbers():
        for id, org_id, uid, type, label, domain, revision in iterate_over_items(shard):
            data[uid].append(Item(id, shard, org_id, type, label, uid, domain, revision))

    # Теперь оставим только рассылки которые дублируются
    data = dict(
        (uid, items)
        for uid, items in list(data.items())
        if len(items) > 1
    )
    return data


# TODO: uncomment before using
# fixed = set()


def get_domain(uid):
    return app.blackbox_instance.userinfo(uid=uid, userip='127.0.0.1')['domain']


def has_user_accounts(domain_name):
    accounts = get_account_list_from_blackbox(domain=domain_name)
    for account in accounts:
        if not account['is_maillist']:
            return True
    return False


def get_user_accounts(domain_name):
    accounts = get_account_list_from_blackbox(domain=domain_name)
    return [a for a in accounts if not a['is_maillist']]


def get_master(domain_name):
    """Вовращает мастер домен. Если domain_name и так является мастер-доменом, то будет возвращён он же.

       Если домена с заданным именем не существует, то возвращается None
    """
    hosted_domains = app.blackbox_instance.hosted_domains(domain=domain_name)['hosted_domains']
    if hosted_domains:
        for domain in hosted_domains:
            if domain['master_domain']:
                return domain['master_domain'].decode('idna')
            else:
                return domain_name


def delete_maillists(domain):
    accounts = get_account_list_from_blackbox(domain=domain)
    for account in accounts:
        uid = account['uid']
        if not account['is_maillist']:
            raise RuntimeError('Unable to delete {} because it is not a maillist'.format(uid))
        app.passport.account_delete(uid)


def make_alias(master, alias):
    master_domain_id = get_domain_id_from_blackbox(master)
    app.passport.domain_alias_add(master_domain_id, alias.encode('idna'))


def delete_domain(domain):
    accounts = get_account_list_from_blackbox(domain=domain)

    if accounts:
        RuntimeError('Unable to delete domain with accounts')

    domain_id = get_domain_id_from_blackbox(domain)
    app.passport.domain_delete(domain_id)


def downgrade_masters_to_aliases(org_id, masters):
    """Превращает все домены кроме одного - в алиасы.
       Так как алиасом можно сделать только домен без учёток, то
       мы пытаемся найти домены без пользователей (если есть рассылки, то это не проблема,
       так как их можно удалить.

       Если находим больше одного домена с учётками, то в этой ситуации мы не можем ничего сделать.
    """
    with_accounts = []
    without_accounts = []

    for domain in masters:
        if has_user_accounts(domain):
            with_accounts.append(domain)
        else:
            without_accounts.append(domain)

    if len(with_accounts) > 1:
        raise RuntimeError('Org {} have more than one master with accounts in passport: {}'.format(
            org_id,
            ', '.join(d.encode('idna') for d in sorted(with_accounts))
        ))

    # Если ни на одном домене нет учётки, то выберем основным первый попавшийся
    if not with_accounts:
        with_accounts = without_accounts[0]
        without_accounts = without_accounts[1:]

    master = with_accounts[0]

    for domain in without_accounts:
        delete_maillists(domain)
        delete_domain(domain)
        make_alias(master, domain)

    return with_accounts, without_accounts



def fix_master_domain(item):
    with get_main_connection(shard=item.shard, for_write=True) as main_connection:
        # Найдём какой домен является мастером в Паспорте
        # и сделаем его мастером в Коннекте

        org_id = item.org_id
        all_domains = DomainModel(main_connection).filter(org_id=org_id, master=True, owned=True).scalar('name')
        all_owned_domains = DomainModel(main_connection).filter(org_id=org_id, owned=True).scalar('name')

        masters = set([_f for _f in map(get_master, all_domains) if _f])

        if len(masters) > 1:
            downgrade_masters_to_aliases(org_id, masters)
        else:
            master = list(masters)[0]
            if master not in all_owned_domains:
                raise RuntimeError('Org {} have no domain {} which is master for other domains'.format(
                    org_id,
                    master.encode('idna'),
                ))

            DomainModel(main_connection).filter(org_id=org_id, master=True).update(master=False)
            DomainModel(main_connection).filter(org_id=org_id, name=master).update(master=True)
            return master


def fix_one_org(uid, items):
    ids = set([(item.org_id, item.id, item.type)
              for item in items])

    # Если все сущности одинаковые, значит в организации
    # несколько подтверждённых мастер-доменов
    if len(items) > 1 and len(ids) == 1:
        # и их надо починить
        first_item = items[0]
        fix_master_domain(first_item)
    else:
        main, other = select_main_for_one_org(items)
        for item in other:
            rename_maillist(item)


def select_main(items, domain):
    main = None
    other = []
    for item in items:
        if item.domain == domain:
            main = item
        else:
            other.append(item)
    return main, other


def select_main_for_one_org(items):
    main = None
    other = []
    for item in items:
        org_id = item.org_id

        if item.type == 'department':
            if main is not None:
                raise RuntimeError('Two departments in same org: {}'.format(org_id))
            main = item
        else:
            other.append(item)

    if main is None:
        raise RuntimeError('No department in org: {}'.format(org_id))
    return main, other



def recreate_maillist(item):
    with get_main_connection(shard=item.shard, for_write=True) as main_connection:
        if item.type == 'department':
            model = DepartmentModel(main_connection)
        else:
            model = GroupModel(main_connection)

        # Может такое быть, что рассылка all на самом деле пользователь
        # или его алиас. В таком случае, для отдела нужно создать другую рассылку
        already_exists = get_user_data_from_blackbox_by_login('{}@{}'.format(item.label, item.domain))

        if already_exists:
            is_user = UserModel(main_connection).filter(
                org_id=item.org_id,
                id=already_exists['uid'],
            ).count() > 0
        else:
            is_user = False

        if is_user:
            rename_maillist(item)
        else:
            if already_exists:
                raise RuntimeError('Account {} already exists in domain {} probably it is an alias from another organization.'.format(
                    item.uid,
                    item.domain,
                ))
            new_uid = create_maillist(
                main_connection,
                item.org_id,
                item.label,
                ignore_login_not_available=True,
            )

            (
                model
                .filter(
                    org_id=item.org_id,
                    label=item.label,
                    uid=item.uid,
                )
                .update(
                    uid=new_uid,
                )
            )

        try:
            MaillistsCheckTask(main_connection).delay(
                org_id=item.org_id,
            )
        except DuplicatedTask:
            # если таск уже есть - не нужно пробовать создать его снова
            pass


def rename_maillist(item):
    """В items, которые в той же организации, переименуем
       рассылку в all-team и пересоздадим её uid.
    """
    with get_main_connection(shard=item.shard, for_write=True) as main_connection:
        if item.type == 'department':
            model = DepartmentModel(main_connection)
        else:
            model = GroupModel(main_connection)

        new_label = item.label + '-team'

        new_uid = create_maillist(
            main_connection,
            item.org_id,
            new_label,
        )

        (
            model
            .filter(
                org_id=item.org_id,
                label=item.label,
                uid=item.uid,
            )
            .update(
                label=new_label,
                uid=new_uid,
            )
        )

        try:
            MaillistsCheckTask(main_connection).delay(
                org_id=item.org_id,
            )
        except DuplicatedTask:
            # если таск уже есть - не нужно пробовать создать его снова
            pass


def turn_wrong_domain_off(item):
   """Отключает домен от организации, удаляя все принадлежащие ему учётки
      и вычищая uid/label из групп и отделов.
   """
   with get_meta_connection(for_write=True) as meta_connection, \
        get_main_connection(shard=item.shard, for_write=True) as main_connection:
      # Сначала получим данные про то, какие учётки есть в домене
      accounts = get_account_list_from_blackbox(domain=item.domain)
      accounts = set(int(a['uid']) for a in accounts)

      org_id = item.org_id

      # Сначала убедимся, что домен всего один, потому что если их больше то
      # это уже более сложная ситуация и её надо как-то по особому обрабатывать.
      domains = DomainModel(main_connection) \
         .filter(org_id=org_id, owned=True) \
         .scalar('name')

      if len(domains) > 1:
         raise RuntimeError('Unable to turn off domain from org_id {}'.format(org_id))


      # Удалим все учётки
      UserModel(main_connection) \
         .filter(org_id=org_id, id=tuple(accounts)) \
         .delete()

      GroupModel(main_connection) \
         .filter(org_id=org_id, uid=tuple(accounts)) \
         .update(label=None, uid=None)

      DepartmentModel(main_connection) \
         .filter(org_id=org_id, uid=tuple(accounts)) \
         .update(label=None, uid=None)

      DomainModel(main_connection) \
         .filter(org_id=org_id, name=item.domain) \
         .update(master=False, owned=False)



def fix_multiple_orgs(uid, items):
    # Иногда учётки в паспорте уже нет, и тогда её надо
    # завести для каждой организации

    wrong_domain_item, move_to_org_id = find_wrong_aliases(items)
    if wrong_domain_item is not None:
        print(('Domain "{}" should be moved to org_id {} and removed from {}'.format(
           wrong_domain_item.domain.encode('idna'),
           move_to_org_id,
           wrong_domain_item.org_id,
        )))
        turn_wrong_domain_off(wrong_domain_item)
        return

    exists = get_user_data_from_blackbox_by_uid(uid)

    if not exists:
        for item in items:
            recreate_maillist(item)
    else:
        # В противном случае, необходимо пересоздать рассылку
        # во всех организациях кроме той, которой принадлежит uid.
        domain = get_domain(uid)
        if not domain:
            raise RuntimeError('No domain for uid {}'.format(uid))

        main_item, other_items = select_main(items, domain)

        for item in other_items:
            recreate_maillist(item)


def fix(uid, items):
    # Сначала надо определить с чем имеем дело, это одна организация или несколько
    org_ids = set([item.org_id for item in items])

    if uid in fixed:
        return

    # Иногда попадаются организации у которых есть рассылки, но нет ни одного домена
    # В таких надо просто удалить uid и label из отделов и команд
    items_without_domain = [
       item
       for item in items
       if not item.domain
    ]

    if items_without_domain:
       for item in items_without_domain:
          clean_uid_and_label(item)
    elif len(org_ids) == 1:
        fix_one_org(uid, items)
    else:
        fix_multiple_orgs(uid, items)

    fixed.add(uid)


def clean_uid_and_label(item):
   assert item.domain is None

   with get_main_connection(shard=item.shard, for_write=True) as main_connection:
      GroupModel(main_connection) \
         .filter(org_id=item.org_id) \
         .update(label=None, uid=None)
      DepartmentModel(main_connection) \
         .filter(org_id=item.org_id) \
         .update(label=None, uid=None)


def remove_from_fixed(data):
    for uid in data:
        fixed.discard(uid)


def process(data=None):
    if data is None:
        data = fetch_data()

    for uid, items in list(data.items()):
        try:
            fix(uid, items)
        except Exception as exc:
            print('ERROR: uid = {} {}'.format(uid, exc))



from intranet.yandex_directory.src.yandex_directory.core.views.organization.view import create_organization_without_domain


def separate_domain(org_id, domain, dry_run=True):
    """Отрывает указанный домен от организации и создаёт новую в том же шарде, и к ней прикрепляет домен и всех сотрудников которые в нём состоят.
       ВНИМАНИЕ! Это опасная и плохо отлаженная процедура, её можно запускать только вручную на одной организации!"""
    with get_meta_connection(for_write=True) as meta_connection:
        shard = get_shard(meta_connection, org_id)
        with get_main_connection(shard=shard, for_write=True) as main_connection:
            accounts = get_account_list_from_blackbox(domain=domain)
            users = [int(a['uid']) for a in accounts if not a['is_maillist']]
            maillists = [int(a['uid']) for a in accounts if a['is_maillist']]

            if len(maillists) > 1:
                # Пока не понятно, что делать если на домене есть команды или отделы помимо all@
                # uid для all@ рассылки можно просто вычистить и он пересоздастся при следующем запуске
                # этого скрипта
                raise RuntimeError('Seems there are more than one all@ maillist')

            print('Я создам отдельную организацию для домена {}'.format(domain))
            print('Затем перенесу в неё пользователей с уидами: {} так как они принадлежат этому домену'.format(
                ', '.join(map(str, users))))
            print('И сброшу эти uid для отделов и команд, в организации {}: {} так как эти рассылки должны быть созданы в новой организации'.format(
                org_id,
                ', '.join(map(str, maillists))))

            if not dry_run:
                # Это нужно для того, чтобы можно было нарушить ссылочную целостность
                # Но чтобы сработало, надо сначала сделать такой ALTER в этом шарде:
                # alter table robot_services alter constraint fk_robot_services_uid DEFERRABLE;
                # http://dbadailystuff.com/deferred-constraints-in-postgresql
                meta_connection.execute('SET CONSTRAINTS ALL DEFERRED')
                main_connection.execute('SET CONSTRAINTS ALL DEFERRED')

                admin_uid = get_organization_admin_uid(main_connection, org_id)

                new_org_id = create_organization_without_domain(
                    meta_connection,
                    main_connection,
                    {},
                    admin_uid,
                    '127.0.0.1',
                )
                print('Создана организация с uid: {}'.format(new_org_id))
                DomainModel(main_connection) \
                    .filter(org_id=org_id, name=domain) \
                    .update(org_id=new_org_id)
                # С членством в группах всё сложно, для того, чтобы их корректно перенести,
                # надо все такие же группы заводить и в новых организациях.
                # Пока что просто удалим их из старой.
                UserGroupMembership(main_connection) \
                    .filter(org_id=org_id,
                            user_id=users) \
                    .delete()

                for user_id in users:
                    ResourceRelationModel(main_connection) \
                        .filter(org_id=org_id, user_id=user_id) \
                        .delete()

                UserModel(main_connection) \
                    .filter(org_id=org_id, id=users) \
                    .update(org_id=new_org_id)

                UserMetaModel(meta_connection) \
                    .filter(org_id=org_id, id=users) \
                    .update(org_id=new_org_id)

                # признаки принадлежности роботов надо перенести
                RobotServiceModel(main_connection) \
                    .filter(org_id=org_id, uid=users) \
                    .update(org_id=new_org_id)


                # В оригинальной организации почистим uid у команд и отделов
                GroupModel(main_connection) \
                    .filter(org_id=org_id, uid=maillists) \
                    .update(uid=None)
                DepartmentModel(main_connection) \
                    .filter(org_id=org_id, uid=maillists) \
                    .update(uid=None)


            return users, maillists


def find_wrong_aliases(items):
    """Эта функция пытается найти ситуации, когда домен, являющийся в Коннекте мастером в организации А
       на самом деле алиас другого домена из организации Б.

       В этом случае домен надо из организации А оторвать (сделать ему owned=False, master=False) и все учётки,
       связанные с ним - удалить. Но только при условии, что в организации A нет других доменов, потому что
       если есть, то это могут быть учётки с них.

       Если такой странный домен найден, то функция вернёт его item и org_id в которой домен так же существует.
       Если со всеми доменами всё в порядке, то вернется просто None, None
    """
    domain_to_org_id = {
        item.domain: item.org_id
        for item in items
    }

    for item in items:
        real_master = get_master(item.domain)
        if real_master != item.domain: # если это алиас
            # и у его мастера другая организация, то мы нашли то что искали
            real_master_org_id = domain_to_org_id.get(real_master.encode('idna'))
            if real_master_org_id:
                if real_master_org_id != item.org_id:
                    return item, real_master_org_id
                else:
                    return None, None
            else:
                raise RuntimeError('Unable to find real master "{}" between items.'.format(real_master.encode('idna')))

    return None, None

