# -*- coding: utf-8 -*-
import warnings
from math import ceil
from functools import wraps
from itertools import chain

from blackbox import (
    JsonBlackbox,
    FIELD_LOGIN,
    FIELD_FIO,
    FIELD_NICKNAME,
    FIELD_SEX,
    FIELD_EMAIL,
    FIELD_COUNTRY,
    FIELD_LANGUAGE,
    FIELD_LOGIN_RULE,
)

FIELD_FIRSTNAME = ('userinfo.firstname.uid', 'first_name')
FIELD_LASTNAME = ('userinfo.lastname.uid', 'last_name')


IS_MAILLIST_ATTRIBUTE = '13'
TIMEZONE_ATTRIBUTE = '33'
LANGUAGE_ATTRIBUTE = '34'
AVATAR_ATTRIBUTE = '98'
IS_AVAILABLE_ATTRIBUTE = '1009'
ORG_IDS_ATTRIBUTE = '1017'
FIRST_NAME_ATTRIBUTE = '27'
LAST_NAME_ATTRIBUTE = '28'
CLOUD_UID_ATTRIBUTE = '193'


def FIELD_SUID(sid):
    """
    Если пользователь подписан на сервис sid такой-то, в ответе
    будет не-None значение в поле suid.

    """
    return ('subscription.suid.%d' % sid, 'suid')


class odict(dict):
    """Обёртка для словаря, добавляет доступ по именам атрибутов."""

    def __getattr__(self, name):
        if name == '__wrapped__':
            raise AttributeError
        try:
            return self[name]
        except KeyError:
            raise AttributeError(name)

    def __setattr__(self, name, value):
        self[name] = value


class BlackboxClient(JsonBlackbox):
    """Класс-посредник, созданный для безболезненного перехода с XML BB на JSON BB."""

    def _blackbox_json_call(self, method, *args, **kwargs):

        def _get_default_email(userinfo_json):
            default_emails = [
                address['address']
                for address in userinfo_json.get('address-list', [])
                if address['default']
            ]
            return default_emails[0] if default_emails else None

        def _get_fields_from_dbfields(userinfo_json):
            fieldname_dict = dict((
                FIELD_LOGIN,
                FIELD_FIO,
                FIELD_NICKNAME,
                FIELD_FIRSTNAME,
                FIELD_LASTNAME,
                FIELD_SEX,
                FIELD_EMAIL,
                FIELD_COUNTRY,
                FIELD_LANGUAGE,
                FIELD_LOGIN_RULE,
            ))

            fields = {
                fieldname_dict.get(
                    field_name,
                    field_name.split('.')[1]
                ): field_value
                for field_name, field_value
                in userinfo_json.get('dbfields', {}).items()
            }

            # alias_type=6 stands for "social alias":
            # https://wiki.yandex-team.ru/passport/dbmoving/#tipyaliasov
            fields['social_aliases'] = [
                alias_value
                for alias_type, alias_value
                in result.get('aliases', {}).items()
                if alias_type == '6'
            ]
            fields['aliases'] = [
                (alias_type, alias_value)
                for alias_type, alias_value
                in result.get('aliases', {}).items()
                if alias_type != '6'
            ]

            return fields

        def _get_domain(userinfo_json):
            uid_json = userinfo_json.get('uid', {})
            if uid_json.get('hosted'):
                return uid_json.get('domain')
            return None

        def _parse_result(raw_json_result):
            parsed_result = dict(
                error=raw_json_result.get('error'),
                exception=raw_json_result.get('exception'),
                uid=raw_json_result.get('uid', {}).get('value'),
                karma=raw_json_result.get('karma', {}).get('value', 0),
                karma_status=raw_json_result.get('karma_status', {}).get('value', 0),
                domain=_get_domain(raw_json_result),
                lite_uid=raw_json_result.get('liteuid'),
                hosted_domains=raw_json_result.get('hosted_domains', []),
                fields=_get_fields_from_dbfields(raw_json_result),
                emails=raw_json_result.get('address-list'),
                default_email=_get_default_email(raw_json_result),
                attributes=raw_json_result.get('attributes', {}),
            )
            parsed_result = {**raw_json_result, **parsed_result}
            return {
                key: (
                    value.get('value')
                    if isinstance(value, dict) and 'value' in value
                    else value
                )
                for key, value
                in parsed_result.items()
            }

        raw_dbfields = []
        if 'dbfields' in kwargs:
            kwargs['dbfields'] = dict(kwargs['dbfields']) if kwargs['dbfields'] else {}
            raw_dbfields = list(kwargs['dbfields'].keys())
            kwargs['dbfields'] = raw_dbfields

        result = super(BlackboxClient, self)._blackbox_json_call(method, *args, **kwargs)

        if 'users' in result:
            if len(result['users']) != 1:
                return [_parse_result(user_json) for user_json in result['users']]
            return _parse_result(result['users'][0])
        return _parse_result(result)

    def _social_params(self, dbfields, social_info, kw):
        if (dbfields and FIELD_LOGIN in dbfields) or social_info:
            kw['regname'] = 'yes'
            kw['aliases'] = 'all'

    def get_uids_by_session(self, request, sessionid, userip, host):
        result = self.sessionid(
            userip=userip,
            sessionid=sessionid,
            host=host,
            request=request,
            dbfields=[],
            multisession=True,
            batch=True,
        )
        if isinstance(result, list):
            return [
                user_info['uid']
                for user_info in result
            ]
        return [result['uid']]

    def oauth(self, headers_or_token, userip, dbfields=None, by_token=False, **kw):
        if by_token:
            result = self._blackbox_json_call(
                'oauth',
                debfields=dbfields,
                oauth_token=headers_or_token,
                userip=userip,
                **kw
            )
        else:
            result = self._blackbox_json_call(
                'oauth',
                debfields=dbfields,
                headers=headers_or_token,
                userip=userip,
                **kw
            )

        return odict(
            valid=(result.get('error') == 'OK'),
            redirect=False,
            new_session=None,
            secure=False,
            new_sslsession=None,
            **result
        )

    def userinfo(self, *args, **kwargs):
        """blackbox.userinfo, мимикрирующий под старый XML-ный формат"""
        if 'attributes' in kwargs and isinstance(kwargs['attributes'], list):
            kwargs['attributes'] = ','.join(kwargs['attributes'])

        userinfo_in_json_format = super(BlackboxClient, self).userinfo(*args, **kwargs)
        if 'users' in userinfo_in_json_format:
            if len(userinfo_in_json_format['users']) > 1:
                return userinfo_in_json_format['users']
            else:
                return userinfo_in_json_format['users'][0]
        else:
            return userinfo_in_json_format

    def batch_userinfo(self, uids, userip='127.0.0.1', dbfields=None,
                       chunk_size=None, social_info=True, **kw):
        """blackbox.userinfo для "пачки" пользователей."""
        from intranet.yandex_directory.src.yandex_directory.core.utils import pmap

        if not isinstance(uids, (list, tuple)):
            raise RuntimeError(
                'Batch operation requires a list or a tuple as first argument'
            )

        self._social_params(dbfields, social_info, kw)

        def get_chunk(uids):
            # API блекбокса так устроено, что если ему передан один элемент,
            # то оно возвращает объект, а если передан список из более чем одного uid,
            # то возвращается список.
            # Тут же мы хотим, чтобы функция всегда возвращала список независимо от того,
            # сколько элементов было в изначальном списке уидов или логинов.
            result = self.userinfo(
                uid=uids,
                dbfields=dbfields or self.dbfields,
                userip=userip,
                **kw
            )

            # Если в списке был всего один uid, то
            # blackbox_json_call вернёт словарь, а не список.
            # Но нам нужно вернуть список в любом случае.
            if not isinstance(result, list):
                result = [result]

            return [odict(**item) for item in result]

        chunk_size = chunk_size or 50
        total_chunks = ceil(len(uids)/chunk_size)
        chunk_results = pmap(
            get_chunk,
            [
                {'uids': uids[chunk_index * chunk_size : (chunk_index + 1) * chunk_size]}
                for chunk_index in range(total_chunks)
            ]
        )
        return list(chain(*chunk_results))

    def hosted_domains(self, **kwargs):
        """
        Метод возвращает выборку из таблицы hosted_domains.
        Требуется наличие гранта <allow_hosted_domains/>
        Параметры:
            domain_id       - id домена
            domain          - имя домена
            domain_admin    - uid администратора домена
            aliases         - булевский флаг, если указан,
            ищем также алиасы домена (выдается в колонке slaves)
        Метод возвращает все строки из таблицы hosted_domains,
        удовлетворяющие всем заданным условиям
        (совпадение по id домена, имени домена и Uid администратора).
        Обязательно наличие в запросе хотя бы одного из этих трех условий.
        """
        result = self._blackbox_json_call('hosted_domains', **kwargs)
        return odict(**result)

    def account_uids(self, **kwargs):
        """
        Возвращает список uid'ов по заданным критериям.
        Обязательно указание либо domain, либо domain_id.

        Дока:
        https://wiki.yandex-team.ru/passport/blackbox/#03.10.2012findpddaccounts
        """

        params = kwargs.copy()
        params['offset'] = 0
        params['limit'] = 100
        results = []

        while True:
            result = self._blackbox_json_call(
                'find_pdd_accounts',
                **params
            )
            total_count = result['total_count']
            results.extend(result['uids'])
            if len(results) == int(total_count):
                break
            else:
                params['offset'] += int(result['count'])

        return [int(user_uid) for user_uid in results]

    def uid(self, login, **kw):
        """Получить UID по логину пользователя, else None."""

        result = self.userinfo(
            login=login,
            userip='127.0.0.1',
            **kw
        )
        try:
            return int(result['uid'])
        except ValueError:
            return None

    def subscription(self, uid, sid):
        """Проверка подписки пользователя на сервис по uid.

        Выдаёт suid и uid пользователя на сервисе, если у пользователя имеется
        подписка на указанный сервис, иначе None.

        """
        result = self.userinfo(
            uid=uid,
            userip='127.0.0.1',
            dbfields=[
                FIELD_SUID(sid),
            ]
        )

        try:
            suid = result['fields']['suid']
            if suid is None:
                return None
            return dict(
                uid=result['uid'],
                suid=result['fields']['suid']
            )
        except KeyError:
            return None

    def sex(self, uid):
        """Пол пользователя по uid

        значение 0 - пол не указан
        значение 1 - мужской
        значение 2 - женский

        """
        result = self.userinfo(
            uid=uid,
            userip='127.0.0.1',
            dbfields=[FIELD_SEX, ]
        )

        try:
            if result['fields']['sex'] is None:
                return None

            return int(result['fields']['sex'])
        except (KeyError, ValueError):
            return None

    def is_lite(self, uid):
        """Проверяет что пользователь с данным uid'ом лайт-пользователь."""

        result = self.userinfo(
            uid=uid,
            userip='127.0.0.1',
            dbfields=[FIELD_LOGIN_RULE],
        )

        return self._check_login_rule(
            result['fields']['login_rule']
        )

    def country(self, uid):
        """Страна регистрации пользователя."""

        result = self.userinfo(
            uid=uid,
            userip='127.0.0.1',
            dbfields=[FIELD_COUNTRY, ]
        )

        try:
            return result['fields']['country']
        except KeyError:
            return None


# Слой совместимости

Blackbox = BlackboxClient

HTTP_TIMEOUT = Blackbox.TIMEOUT

BLACKBOX_URL_PRODUCTION = Blackbox.URLS['other']['production']

BLACKBOX_URL_DEVELOPMENT = Blackbox.URLS['other']['development']

BLACKBOX_URL = Blackbox.URL


def caller(name):
    @wraps(getattr(Blackbox, name))
    def _decorator(*args, **kwargs):
        if BLACKBOX_URL != Blackbox.URL or HTTP_TIMEOUT != Blackbox.TIMEOUT:
            warnings.warn(
                'You should not monkey-patch blackbox module! '
                'Use BlackboxClient class instead.',
                DeprecationWarning
            )
        # Инстанцируем объект Blackbox с целью эмуляции старого поведения
        return getattr(Blackbox(BLACKBOX_URL, HTTP_TIMEOUT), name)(*args, **kwargs)

    return _decorator


sessionid = caller('sessionid')
login = caller('login')
userinfo = caller('userinfo')
hosted_domains = caller('hosted_domains')
uid = caller('uid')
subscription = caller('subscription')
sex = caller('sex')
is_lite = caller('is_lite')
oauth = caller('oauth')
country = caller('country')
_check_login_rule = caller('_check_login_rule')
